const path = require("path");
const copy = require("copy-webpack-plugin");
const extract = require("extract-text-webpack-plugin");
const fs = require("fs");
const webpack = require("webpack");
const CompressionPlugin = require("compression-webpack-plugin");

var externals = {
    "cockpit": "cockpit",
};

/* These can be overridden, typically from the Makefile.am */
const srcdir = (process.env.SRCDIR || __dirname) + path.sep + "src";
const builddir = (process.env.SRCDIR || __dirname);
const distdir = builddir + path.sep + "dist";
const section = process.env.ONLYDIR || null;
const libdir = path.resolve(srcdir, "pkg" + path.sep + "lib");
const nodedir = path.resolve((process.env.SRCDIR || __dirname), "node_modules");

/* A standard nodejs and webpack pattern */
var production = process.env.NODE_ENV === 'production';

var info = {
    entries: {
        "recordings": [
            "./recordings.jsx",
            "./recordings.css",
            "./pkg/lib/listing.less",
        ],
        "config": [
            "./config.jsx",
            "./recordings.css",
            "./table.css",
        ]
    },
    files: [
        "index.html",
        "config.html",
        "player.jsx",
        "recordings.jsx",
        "recordings.css",
        "table.css",
        "terminal.jsx",
        "manifest.json",
        "timer.css",
        "./pkg/lib/listing.less",
    ],
};

var output = {
    path: distdir,
    filename: "[name].js",
    sourceMapFilename: "[file].map",
};

/*
 * Note that we're avoiding the use of path.join as webpack and nodejs
 * want relative paths that start with ./ explicitly.
 *
 * In addition we mimic the VPATH style functionality of GNU Makefile
 * where we first check builddir, and then srcdir.
 */

function vpath(/* ... */) {
    var filename = Array.prototype.join.call(arguments, path.sep);
    var expanded = builddir + path.sep + filename;
    if (fs.existsSync(expanded))
        return expanded;
    expanded = srcdir + path.sep + filename;
    return expanded;
}

/* Qualify all the paths in entries */
Object.keys(info.entries).forEach(function(key) {
    if (section && key.indexOf(section) !== 0) {
        delete info.entries[key];
        return;
    }

    info.entries[key] = info.entries[key].map(function(value) {
        if (value.indexOf("/") === -1)
            return value;
        else
            return vpath(value);
    });
});

/* Qualify all the paths in files listed */
var files = [];
info.files.forEach(function(value) {
    if (!section || value.indexOf(section) === 0)
        files.push({ from: vpath("src", value), to: value });
});
info.files = files;

var plugins = [
    new copy(info.files),
    new extract("[name].css")
];

/* Only minimize when in production mode */
if (production) {
    /* Rename output files when minimizing */
    output.filename = "[name].min.js";

    plugins.unshift(new CompressionPlugin({
        asset: "[path].gz[query]",
        test: /\.(js|html)$/,
        minRatio: 0.9,
        deleteOriginalAssets: true
    }));
}

module.exports = {
    mode: production ? 'production' : 'development',
    entry: info.entries,
    externals: externals,
    output: output,
    devtool: "source-map",
    resolve: {
        alias: {
            "fs": path.resolve(nodedir, "fs-extra"),
        },
        modules: [libdir, nodedir],
    },
    module: {
        rules: [
            {
                enforce: 'pre',
                exclude: /node_modules/,
                loader: 'eslint-loader',
                test: /\.jsx$/
            },
            {
                enforce: 'pre',
                exclude: /node_modules/,
                loader: 'eslint-loader',
                test: /\.es6$/
            },
            {
                exclude: /node_modules/,
                loader: 'babel-loader',
                test: /\.js$/
            },
            {
                exclude: /node_modules/,
                loader: 'babel-loader',
                test: /\.jsx$/
            },
            {
                exclude: /node_modules/,
                loader: 'babel-loader',
                test: /\.es6$/
            },
            {
                test: /\.less$/,
                loader: extract.extract("css-loader!less-loader")
            },
            {
                exclude: /node_modules/,
                loader: extract.extract('css-loader!sass-loader'),
                test: /\.scss$/
            },
            {
                loader: extract.extract("css-loader?minimize=&root=" + libdir),
                test: /\.css$/,
            }
        ]
    },
    plugins: plugins
}
