﻿// Copyright (c) Microsoft.  All Rights Reserved.  Licensed under the Apache License, Version 2.0.  See License.txt in the project root for license information.

using Microsoft.CodeAnalysis.Test.Utilities;
using Xunit;

namespace Microsoft.CodeAnalysis.Editor.UnitTests.CodeGeneration
{
    [Trait(Traits.Feature, Traits.Features.CodeGeneration)]
    public class ExpressionGenerationTests : AbstractCodeGenerationTests
    {
        [Fact]
        public void TestFalseExpression()
        {
            Test(
                f => f.FalseLiteralExpression(),
                cs: "false",
                vb: "False");
        }

        [Fact]
        public void TestTrueExpression()
        {
            Test(
                f => f.TrueLiteralExpression(),
                cs: "true",
                vb: "True");
        }

        [Fact]
        public void TestNullExpression()
        {
            Test(
                f => f.NullLiteralExpression(),
                cs: "null",
                vb: "Nothing");
        }

        [Fact]
        public void TestThisExpression()
        {
            Test(
                f => f.ThisExpression(),
                cs: "this",
                vb: "Me");
        }

        [Fact]
        public void TestBaseExpression()
        {
            Test(
                f => f.BaseExpression(),
                cs: "base",
                vb: "MyBase");
        }

        [Fact]
        public void TestInt32LiteralExpression0()
        {
            Test(
                f => f.LiteralExpression(0),
                cs: "0",
                vb: "0");
        }

        [Fact]
        public void TestInt32LiteralExpression1()
        {
            Test(
                f => f.LiteralExpression(1),
                cs: "1",
                vb: "1");
        }

        [Fact]
        public void TestInt64LiteralExpression0()
        {
            Test(
                f => f.LiteralExpression(0L),
                cs: "0L",
                vb: "0L");
        }

        [Fact]
        public void TestInt64LiteralExpression1()
        {
            Test(
                f => f.LiteralExpression(1L),
                cs: "1L",
                vb: "1L");
        }

        [Fact]
        public void TestSingleLiteralExpression0()
        {
            Test(
                f => f.LiteralExpression(0.0f),
                cs: "0F",
                vb: "0F");
        }

        [Fact]
        public void TestSingleLiteralExpression1()
        {
            Test(
                f => f.LiteralExpression(0.5F),
                cs: "0.5F",
                vb: "0.5F");
        }

        [Fact]
        public void TestDoubleLiteralExpression0()
        {
            Test(
                f => f.LiteralExpression(0.0d),
                cs: "0D",
                vb: "0R");
        }

        [Fact]
        public void TestDoubleLiteralExpression1()
        {
            Test(
                f => f.LiteralExpression(0.5D),
                cs: "0.5D",
                vb: "0.5R");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestAddExpression1()
        {
            Test(
                f => f.AddExpression(
                    f.LiteralExpression(1),
                    f.LiteralExpression(2)),
                cs: "1 + 2",
                vb: "1 + 2");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestAddExpression2()
        {
            Test(
                f => f.AddExpression(
                    f.LiteralExpression(1),
                    f.AddExpression(
                        f.LiteralExpression(2),
                        f.LiteralExpression(3))),
                cs: "1 + 2 + 3",
                vb: "1 + 2 + 3");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestAddExpression3()
        {
            Test(
                f => f.AddExpression(
                    f.AddExpression(
                        f.LiteralExpression(1),
                        f.LiteralExpression(2)),
                    f.LiteralExpression(3)),
                cs: "1 + 2 + 3",
                vb: "1 + 2 + 3");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestMultiplyExpression1()
        {
            Test(
                f => f.MultiplyExpression(
                    f.LiteralExpression(1),
                    f.LiteralExpression(2)),
                cs: "1 * 2",
                vb: "1 * 2");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestMultiplyExpression2()
        {
            Test(
                f => f.MultiplyExpression(
                    f.LiteralExpression(1),
                    f.MultiplyExpression(
                        f.LiteralExpression(2),
                        f.LiteralExpression(3))),
                cs: "1 * 2 * 3",
                vb: "1 * 2 * 3");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestMultiplyExpression3()
        {
            Test(
                f => f.MultiplyExpression(
                    f.MultiplyExpression(
                        f.LiteralExpression(1),
                        f.LiteralExpression(2)),
                    f.LiteralExpression(3)),
                cs: "1 * 2 * 3",
                vb: "1 * 2 * 3");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestBinaryAndExpression1()
        {
            Test(
                f => f.BitwiseAndExpression(
                    f.LiteralExpression(1),
                    f.LiteralExpression(2)),
                cs: "1 & 2",
                vb: "1 And 2");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestBinaryOrExpression1()
        {
            Test(
                f => f.BitwiseOrExpression(
                    f.LiteralExpression(1),
                    f.LiteralExpression(2)),
                cs: "1 | 2",
                vb: "1 Or 2");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestLogicalAndExpression1()
        {
            Test(
                f => f.LogicalAndExpression(
                    f.LiteralExpression(1),
                    f.LiteralExpression(2)),
                cs: "1 && 2",
                vb: "1 AndAlso 2");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestLogicalOrExpression1()
        {
            Test(
                f => f.LogicalOrExpression(
                    f.LiteralExpression(1),
                    f.LiteralExpression(2)),
                cs: "1 || 2",
                vb: "1 OrElse 2");
        }

        [Fact]
        public void TestMemberAccess1()
        {
            Test(
                f => f.MemberAccessExpression(
                    f.IdentifierName("E"),
                    f.IdentifierName("M")),
                cs: "E.M",
                vb: "E.M");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestConditionalExpression1()
        {
            Test(
                f => f.ConditionalExpression(
                    f.IdentifierName("E"),
                    f.IdentifierName("T"),
                    f.IdentifierName("F")),
                cs: "E ? T : F",
                vb: "If(E, T, F)");
        }

        [Fact]
        public void TestInvocation1()
        {
            Test(
                f => f.InvocationExpression(
                    f.IdentifierName("E")),
                cs: "E()",
                vb: "E()");
        }

        [Fact]
        public void TestInvocation2()
        {
            Test(
                f => f.InvocationExpression(
                    f.IdentifierName("E"),
                    f.Argument(f.IdentifierName("a"))),
                cs: "E(a)",
                vb: "E(a)");
        }

        [Fact]
        public void TestInvocation3()
        {
            Test(
                f => f.InvocationExpression(
                    f.IdentifierName("E"),
                    f.Argument("n", RefKind.None, f.IdentifierName("a"))),
                cs: "E(n: a)",
                vb: "E(n:=a)");
        }

        [Fact]
        public void TestInvocation4()
        {
            Test(
                f => f.InvocationExpression(
                    f.IdentifierName("E"),
                    f.Argument(null, RefKind.Out, f.IdentifierName("a")),
                    f.Argument(null, RefKind.Ref, f.IdentifierName("b"))),
                cs: "E(out a, ref b)",
                vb: "E(a, b)");
        }

        [Fact]
        public void TestInvocation5()
        {
            Test(
                f => f.InvocationExpression(
                    f.IdentifierName("E"),
                    f.Argument("n1", RefKind.Out, f.IdentifierName("a")),
                    f.Argument("n2", RefKind.Ref, f.IdentifierName("b"))),
                cs: "E(n1: out a, n2: ref b)",
                vb: "E(n1:=a, n2:=b)");
        }

        [Fact]
        public void TestElementAccess1()
        {
            Test(
                f => f.ElementAccessExpression(
                    f.IdentifierName("E")),
                cs: "E[]",
                vb: "E()");
        }

        [Fact]
        public void TestElementAccess2()
        {
            Test(
                f => f.ElementAccessExpression(
                    f.IdentifierName("E"),
                    f.Argument(f.IdentifierName("a"))),
                cs: "E[a]",
                vb: "E(a)");
        }

        [Fact]
        public void TestElementAccess3()
        {
            Test(
                f => f.ElementAccessExpression(
                    f.IdentifierName("E"),
                    f.Argument("n", RefKind.None, f.IdentifierName("a"))),
                cs: "E[n: a]",
                vb: "E(n:=a)");
        }

        [Fact]
        public void TestElementAccess4()
        {
            Test(
                f => f.ElementAccessExpression(
                    f.IdentifierName("E"),
                    f.Argument(null, RefKind.Out, f.IdentifierName("a")),
                    f.Argument(null, RefKind.Ref, f.IdentifierName("b"))),
                cs: "E[out a, ref b]",
                vb: "E(a, b)");
        }

        [Fact]
        public void TestElementAccess5()
        {
            Test(
                f => f.ElementAccessExpression(
                    f.IdentifierName("E"),
                    f.Argument("n1", RefKind.Out, f.IdentifierName("a")),
                    f.Argument("n2", RefKind.Ref, f.IdentifierName("b"))),
                cs: "E[n1: out a, n2: ref b]",
                vb: "E(n1:=a, n2:=b)");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestIsExpression()
        {
            Test(
                f => f.IsTypeExpression(
                    f.IdentifierName("a"),
                    CreateClass("SomeType")),
                cs: "a is SomeType",
                vb: "TypeOf a Is SomeType");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestAsExpression()
        {
            Test(
                f => f.TryCastExpression(
                    f.IdentifierName("a"),
                    CreateClass("SomeType")),
                cs: "a as SomeType",
                vb: "TryCast(a, SomeType)");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestNotExpression()
        {
            Test(
                f => f.LogicalNotExpression(
                    f.IdentifierName("a")),
                cs: "!a",
                vb: "Not a");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestCastExpression()
        {
            Test(
                f => f.CastExpression(
                    CreateClass("SomeType"),
                    f.IdentifierName("a")),
                cs: "(SomeType)a",
                vb: "DirectCast(a, SomeType)");
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/26586")]
        public void TestNegateExpression()
        {
            Test(
                f => f.NegateExpression(
                    f.IdentifierName("a")),
                cs: "-a",
                vb: "-a");
        }
    }
}
