/*
 * Copyright (C) 2018-2019 Red Hat, Inc.  All rights reserved.
 *
 * Author: Fabio M. Di Nitto <fabbione@kronosnet.org>
 *
 * This software licensed under GPL-2.0+
 */

#include "config.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <stdint.h>
#include <limits.h>
#include <sys/socket.h>

#include "test-common.h"

static int test(void)
{
	char verifycmd[1024];
	char device_name[IFNAMSIZ];
	size_t size = IFNAMSIZ;
	int err=0;
	nozzle_t nozzle;
	char *error_string = NULL;

	printf("Testing interface up/down\n");

	memset(device_name, 0, size);
	nozzle = nozzle_open(device_name, size, NULL);
	if (!nozzle) {
		printf("Unable to init %s\n", device_name);
		return -1;
	}

	printf("Put the interface up\n");

	if (nozzle_set_up(nozzle) < 0) {
		printf("Unable to set interface up\n");
		err = -1;
		goto out_clean;
	}

	memset(verifycmd, 0, sizeof(verifycmd));
	snprintf(verifycmd, sizeof(verifycmd)-1,
#ifdef KNET_LINUX
		 "ip addr show dev %s | grep -q UP", nozzle->name);
#endif
#ifdef KNET_BSD
		 "ifconfig %s | grep -q UP", nozzle->name);
#endif
	err = execute_bin_sh_command(verifycmd, &error_string);
	if (error_string) {
		printf("Error string: %s\n", error_string);
		free(error_string);
		error_string = NULL;
	}
	if (err < 0) {
		printf("Unable to verify inteface UP\n");
		err = -1;
		goto out_clean;
	}

	printf("Test ERROR conditions\n");

	printf("Try to UP the same interface twice\n");
	if (nozzle_set_up(nozzle) < 0) {
		printf("Interface was already UP, spurious error received from nozzle_set_up\n");
		err = -1;
		goto out_clean;
	}

	printf("Pass NULL to nozzle set_up\n");
	errno = 0;
	if ((nozzle_set_up(NULL) >= 0) || (errno != EINVAL)) {
		printf("Something is wrong in nozzle_set_up sanity checks\n");
		err = -1;
		goto out_clean;
	}

out_clean:
	nozzle_set_down(nozzle);
	nozzle_close(nozzle);

	return err;
}

int main(void)
{
	need_root();

	if (test() < 0)
		return FAIL;

	return PASS;
}
