# NBD client library in userspace
# WARNING: THIS FILE IS GENERATED FROM
# generator/generator
# ANY CHANGES YOU MAKE TO THIS FILE WILL BE LOST.
#
# Copyright (C) 2013-2021 Red Hat Inc.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

'''
Python bindings for libnbd

import nbd
h = nbd.NBD()
h.connect_tcp("localhost", "nbd")
buf = h.pread(512, 0)

Read the libnbd(3) man page to find out how to use the API.
'''

import libnbdmod

# Re-export Error exception as nbd.Error, adding some methods.
from libnbdmod import Error

Error.__doc__ = '''
Exception thrown when the underlying libnbd call fails.

This exception has three properties to query the error.  Use
the .string property to return a printable string containing
the error message.  Use the .errnum property for the associated
numeric error value (which may be 0 if the error did not
correspond to a system call failure), or the .errno property to
return a string containing the Python errno name if one is known
(which may be None if the numeric value does not correspond to
a known errno name).
'''

Error.string = property(lambda self: self.args[0])


def _errno(self):
    import errno
    try:
        return errno.errorcode[self.args[1]]
    except KeyError:
        return None


Error.errno = property(_errno)

Error.errnum = property(lambda self: self.args[1])


def _str(self):
    if self.errno:
        return "%s (%s)" % (self.string, self.errno)
    else:
        return "%s" % self.string


Error.__str__ = _str

TLS_DISABLE = 0
TLS_ALLOW = 1
TLS_REQUIRE = 2

SIZE_MINIMUM = 0
SIZE_PREFERRED = 1
SIZE_MAXIMUM = 2

CMD_FLAG_FUA = 0x01
CMD_FLAG_NO_HOLE = 0x02
CMD_FLAG_DF = 0x04
CMD_FLAG_REQ_ONE = 0x08
CMD_FLAG_FAST_ZERO = 0x10
CMD_FLAG_MASK = 0x1f

HANDSHAKE_FLAG_FIXED_NEWSTYLE = 0x01
HANDSHAKE_FLAG_NO_ZEROES = 0x02
HANDSHAKE_FLAG_MASK = 0x03

STRICT_COMMANDS = 0x01
STRICT_FLAGS = 0x02
STRICT_BOUNDS = 0x04
STRICT_ZERO_SIZE = 0x08
STRICT_ALIGN = 0x10
STRICT_MASK = 0x1f

ALLOW_TRANSPORT_TCP = 0x01
ALLOW_TRANSPORT_UNIX = 0x02
ALLOW_TRANSPORT_VSOCK = 0x04
ALLOW_TRANSPORT_MASK = 0x07

SHUTDOWN_ABANDON_PENDING = 0x10000
SHUTDOWN_MASK = 0x10000

AIO_DIRECTION_READ = 1
AIO_DIRECTION_WRITE = 2
AIO_DIRECTION_BOTH = 3
READ_DATA = 1
READ_HOLE = 2
READ_ERROR = 3
NAMESPACE_BASE = "base:"
CONTEXT_BASE_ALLOCATION = "base:allocation"
STATE_HOLE = 1
STATE_ZERO = 2


class Buffer(object):
    '''Asynchronous I/O persistent buffer'''

    def __init__(self, len):
        '''allocate an uninitialized AIO buffer used for nbd.aio_pread'''
        self._o = libnbdmod.alloc_aio_buffer(len)

    @classmethod
    def from_bytearray(cls, ba):
        '''create an AIO buffer from a bytearray'''
        o = libnbdmod.aio_buffer_from_bytearray(ba)
        self = cls(0)
        self._o = o
        return self

    def to_bytearray(self):
        '''copy an AIO buffer into a bytearray'''
        return libnbdmod.aio_buffer_to_bytearray(self._o)

    def size(self):
        '''return the size of an AIO buffer'''
        return libnbdmod.aio_buffer_size(self._o)

    def is_zero(self, offset=0, size=-1):
        '''
        Returns true if and only if all bytes in the buffer are zeroes.
        Note that a freshly allocated buffer is uninitialized, not zero.

        By default this tests the whole buffer, but you can restrict
        the test to a sub-range of the buffer using the optional
        offset and size parameters.  If size = -1 then we check from
        offset to the end of the buffer.  If size = 0, the function
        always returns true.  If size > 0, we check the interval
        [offset..offset+size-1].
        '''
        return libnbdmod.aio_buffer_is_zero(self._o, offset, size)


class NBD(object):
    '''NBD handle'''

    def __init__(self):
        '''create a new NBD handle'''
        self._o = libnbdmod.create()

    def __del__(self):
        '''close the NBD handle and underlying connection'''
        if hasattr(self, '_o'):
            libnbdmod.close(self._o)

    def set_debug(self, debug):
        '''▶ set or clear the debug flag

    Set or clear the debug flag. When debugging is enabled,
    debugging messages from the library are printed to
    stderr, unless a debugging callback has been defined too
    (see "nbd.set_debug_callback") in which case they are
    sent to that function. This flag defaults to false on
    newly created handles, except if "DEBUG=1" is set in the
    environment in which case it defaults to true.
'''
        return libnbdmod.set_debug(self._o, debug)

    def get_debug(self):
        '''▶ return the state of the debug flag

    Return the state of the debug flag on this handle.
'''
        return libnbdmod.get_debug(self._o)

    def set_debug_callback(self, debug):
        '''▶ set the debug callback

    Set the debug callback. This function is called when the
    library emits debug messages, when debugging is enabled
    on a handle. The callback parameters are "user_data"
    passed to this function, the name of the libnbd function
    emitting the debug message ("context"), and the message
    itself ("msg"). If no debug callback is set on a handle
    then messages are printed on "stderr".

    The callback should not call "nbd_*" APIs on the same
    handle since it can be called while holding the handle
    lock and will cause a deadlock.
'''
        return libnbdmod.set_debug_callback(self._o, debug)

    def clear_debug_callback(self):
        '''▶ clear the debug callback

    Remove the debug callback if one was previously
    associated with the handle (with
    "nbd.set_debug_callback"). If no callback was associated
    this does nothing.
'''
        return libnbdmod.clear_debug_callback(self._o)

    def set_handle_name(self, handle_name):
        '''▶ set the handle name

    Handles have a name which is unique within the current
    process. The handle name is used in debug output.

    Handle names are normally generated automatically and
    have the form "nbd1", "nbd2", etc., but you can
    optionally use this call to give the handles a name
    which is meaningful for your application to make
    debugging output easier to understand.
'''
        return libnbdmod.set_handle_name(self._o, handle_name)

    def get_handle_name(self):
        '''▶ get the handle name

    Get the name of the handle. If it was previously set by
    calling "nbd.set_handle_name" then this returns the name
    that was set. Otherwise it will return a generic name
    like "nbd1", "nbd2", etc.
'''
        return libnbdmod.get_handle_name(self._o)

    def set_private_data(self, private_data):
        '''▶ set the per-handle private data

    Handles contain a private data field for applications to
    use for any purpose.

    When calling libnbd from C, the type of this field is
    "uintptr_t" so it can be used to store an unsigned
    integer or a pointer.

    In non-C bindings it can be used to store an unsigned
    integer.

    This function sets the value of this field and returns
    the old value (or 0 if it was not previously set).
'''
        return libnbdmod.set_private_data(self._o, private_data)

    def get_private_data(self):
        '''▶ get the per-handle private data

    Return the value of the private data field set
    previously by a call to "nbd.set_private_data" (or 0 if
    it was not previously set).
'''
        return libnbdmod.get_private_data(self._o)

    def set_export_name(self, export_name):
        '''▶ set the export name

    For servers which require an export name or can serve
    different content on different exports, set the
    "export_name" to connect to. The default is the empty
    string "".

    This is only relevant when connecting to servers using
    the newstyle protocol as the oldstyle protocol did not
    support export names. The NBD protocol limits export
    names to 4096 bytes, but servers may not support the
    full length. The encoding of export names is always
    UTF-8.

    When option mode is not in use, the export name must be
    set before beginning a connection. However, when
    "nbd.set_opt_mode" has enabled option mode, it is
    possible to change the export name prior to
    "nbd.opt_go". In particular, the use of "nbd.opt_list"
    during negotiation can be used to determine a name the
    server is likely to accept, and "nbd.opt_info" can be
    used to learn details about an export before connecting.

    This call may be skipped if using "nbd.connect_uri" to
    connect to a URI that includes an export name.
'''
        return libnbdmod.set_export_name(self._o, export_name)

    def get_export_name(self):
        '''▶ get the export name

    Get the export name associated with the handle. This is
    the name that libnbd requests; see
    "nbd.get_canonical_export_name" for determining if the
    server has a different canonical name for the given
    export (most common when requesting the default export
    name of an empty string "")
'''
        return libnbdmod.get_export_name(self._o)

    def set_full_info(self, request):
        '''▶ control whether NBD_OPT_GO requests extra details

    By default, when connecting to an export, libnbd only
    requests the details it needs to service data
    operations. The NBD protocol says that a server can
    supply optional information, such as a canonical name of
    the export (see "nbd.get_canonical_export_name") or a
    description of the export (see
    "nbd.get_export_description"), but that a hint from the
    client makes it more likely for this extra information
    to be provided. This function controls whether libnbd
    will provide that hint.

    Note that even when full info is requested, the server
    is not obligated to reply with all information that
    libnbd requested. Similarly, libnbd will ignore any
    optional server information that libnbd has not yet been
    taught to recognize.
'''
        return libnbdmod.set_full_info(self._o, request)

    def get_full_info(self):
        '''▶ see if NBD_OPT_GO requests extra details

    Return the state of the full info request flag on this
    handle.
'''
        return libnbdmod.get_full_info(self._o)

    def get_canonical_export_name(self):
        '''▶ return the canonical export name, if the server has one

    The NBD protocol permits a server to report an optional
    canonical export name, which may differ from the
    client's request (as set by "nbd.set_export_name" or
    "nbd.connect_uri"). This function accesses any name
    returned by the server; it may be the same as the client
    request, but is more likely to differ when the client
    requested a connection to the default export name (an
    empty string "").

    Some servers are unlikely to report a canonical name
    unless the client specifically hinted about wanting it,
    via "nbd.set_full_info".
'''
        return libnbdmod.get_canonical_export_name(self._o)

    def get_export_description(self):
        '''▶ return the export description, if the server has one

    The NBD protocol permits a server to report an optional
    export description. This function reports any
    description returned by the server.

    Some servers are unlikely to report a description unless
    the client specifically hinted about wanting it, via
    "nbd.set_full_info". For qemu-nbd(8), a description is
    set with *-D*.
'''
        return libnbdmod.get_export_description(self._o)

    def set_tls(self, tls):
        '''▶ enable or require TLS (authentication and encryption)

    Enable or require TLS (authenticated and encrypted
    connections) to the NBD server. The possible settings
    are:

    "TLS_DISABLE"
        Disable TLS. (The default setting, unless using
        "nbd.connect_uri" with a URI that requires TLS)

    "TLS_ALLOW"
        Enable TLS if possible.

        This option is insecure (or best effort) in that in
        some cases it will fall back to an unencrypted
        and/or unauthenticated connection if TLS could not
        be established. Use "TLS_REQUIRE" below if the
        connection must be encrypted.

        Some servers will drop the connection if TLS fails
        so fallback may not be possible.

    "TLS_REQUIRE"
        Require an encrypted and authenticated TLS
        connection. Always fail to connect if the connection
        is not encrypted and authenticated.

    As well as calling this you may also need to supply the
    path to the certificates directory
    ("nbd.set_tls_certificates"), the username
    ("nbd.set_tls_username") and/or the Pre-Shared Keys
    (PSK) file ("nbd.set_tls_psk_file"). For now, when using
    "nbd.connect_uri", any URI query parameters related to
    TLS are not handled automatically. Setting the level
    higher than zero will fail if libnbd was not compiled
    against gnutls; you can test whether this is the case
    with "nbd.supports_tls".
'''
        return libnbdmod.set_tls(self._o, tls)

    def get_tls(self):
        '''▶ get the TLS request setting

    Get the TLS request setting.

    Note: If you want to find out if TLS was actually
    negotiated on a particular connection use
    "nbd.get_tls_negotiated" instead.
'''
        return libnbdmod.get_tls(self._o)

    def get_tls_negotiated(self):
        '''▶ find out if TLS was negotiated on a connection

    After connecting you may call this to find out if the
    connection is using TLS.

    This is only really useful if you set the TLS request
    mode to "TLS_ALLOW" (see "nbd.set_tls"), because in this
    mode we try to use TLS but fall back to unencrypted if
    it was not available. This function will tell you if TLS
    was negotiated or not.

    In "TLS_REQUIRE" mode (the most secure) the connection
    would have failed if TLS could not be negotiated, and in
    "TLS_DISABLE" mode TLS is not tried.
'''
        return libnbdmod.get_tls_negotiated(self._o)

    def set_tls_certificates(self, dir):
        '''▶ set the path to the TLS certificates directory

    Set the path to the TLS certificates directory. If not
    set and TLS is used then a compiled in default is used.
    For root this is "/etc/pki/libnbd/". For non-root this
    is "$HOME/.pki/libnbd" and "$HOME/.config/pki/libnbd".
    If none of these directories can be found then the
    system trusted CAs are used.

    This function may be called regardless of whether TLS is
    supported, but will have no effect unless "nbd.set_tls"
    is also used to request or require TLS.
'''
        return libnbdmod.set_tls_certificates(self._o, dir)

    def set_tls_verify_peer(self, verify):
        '''▶ set whether we verify the identity of the server

    Set this flag to control whether libnbd will verify the
    identity of the server from the server's certificate and
    the certificate authority. This defaults to true when
    connecting to TCP servers using TLS certificate
    authentication, and false otherwise.

    This function may be called regardless of whether TLS is
    supported, but will have no effect unless "nbd.set_tls"
    is also used to request or require TLS.
'''
        return libnbdmod.set_tls_verify_peer(self._o, verify)

    def get_tls_verify_peer(self):
        '''▶ get whether we verify the identity of the server

    Get the verify peer flag.
'''
        return libnbdmod.get_tls_verify_peer(self._o)

    def set_tls_username(self, username):
        '''▶ set the TLS username

    Set the TLS client username. This is used if
    authenticating with PSK over TLS is enabled. If not set
    then the local username is used.

    This function may be called regardless of whether TLS is
    supported, but will have no effect unless "nbd.set_tls"
    is also used to request or require TLS.
'''
        return libnbdmod.set_tls_username(self._o, username)

    def get_tls_username(self):
        '''▶ get the current TLS username

    Get the current TLS username.
'''
        return libnbdmod.get_tls_username(self._o)

    def set_tls_psk_file(self, filename):
        '''▶ set the TLS Pre-Shared Keys (PSK) filename

    Set the TLS Pre-Shared Keys (PSK) filename. This is used
    if trying to authenticate to the server using with a
    pre-shared key. There is no default so if this is not
    set then PSK authentication cannot be used to connect to
    the server.

    This function may be called regardless of whether TLS is
    supported, but will have no effect unless "nbd.set_tls"
    is also used to request or require TLS.
'''
        return libnbdmod.set_tls_psk_file(self._o, filename)

    def set_request_structured_replies(self, request):
        '''▶ control use of structured replies

    By default, libnbd tries to negotiate structured replies
    with the server, as this protocol extension must be in
    use before "nbd.can_meta_context" or "nbd.can_df" can
    return true. However, for integration testing, it can be
    useful to clear this flag rather than find a way to
    alter the server to fail the negotiation request.
'''
        return libnbdmod.set_request_structured_replies(self._o, request)

    def get_request_structured_replies(self):
        '''▶ see if structured replies are attempted

    Return the state of the request structured replies flag
    on this handle.

    Note: If you want to find out if structured replies were
    actually negotiated on a particular connection use
    "nbd.get_structured_replies_negotiated" instead.
'''
        return libnbdmod.get_request_structured_replies(self._o)

    def get_structured_replies_negotiated(self):
        '''▶ see if structured replies are in use

    After connecting you may call this to find out if the
    connection is using structured replies.
'''
        return libnbdmod.get_structured_replies_negotiated(self._o)

    def set_handshake_flags(self, flags):
        '''▶ control use of handshake flags

    By default, libnbd tries to negotiate all possible
    handshake flags that are also supported by the server,
    since omitting a handshake flag can prevent the use of
    other functionality such as TLS encryption or structured
    replies. However, for integration testing, it can be
    useful to reduce the set of flags supported by the
    client to test that a particular server can handle
    various clients that were compliant to older versions of
    the NBD specification.

    The "flags" argument is a bitmask, including zero or
    more of the following handshake flags:

    "HANDSHAKE_FLAG_FIXED_NEWSTYLE" = 1
        The server gracefully handles unknown option
        requests from the client, rather than disconnecting.
        Without this flag, a client cannot safely request to
        use extensions such as TLS encryption or structured
        replies, as the request may cause an older server to
        drop the connection.

    "HANDSHAKE_FLAG_NO_ZEROES" = 2
        If the client is forced to use "NBD_OPT_EXPORT_NAME"
        instead of the preferred "NBD_OPT_GO", this flag
        allows the server to send fewer all-zero padding
        bytes over the connection.

    For convenience, the constant "HANDSHAKE_FLAG_MASK" is
    available to describe all flags supported by this build
    of libnbd. Future NBD extensions may add further flags,
    which in turn may be enabled by default in newer libnbd.
    As such, when attempting to disable only one specific
    bit, it is wiser to first call "nbd.get_handshake_flags"
    and modify that value, rather than blindly setting a
    constant value.
'''
        return libnbdmod.set_handshake_flags(self._o, flags)

    def get_handshake_flags(self):
        '''▶ see which handshake flags are supported

    Return the state of the handshake flags on this handle.
    When the handle has not yet completed a connection (see
    "nbd.aio_is_created"), this returns the flags that the
    client is willing to use, provided the server also
    advertises those flags. After the connection is ready
    (see "nbd.aio_is_ready"), this returns the flags that
    were actually agreed on between the server and client.
    If the NBD protocol defines new handshake flags, then
    the return value from a newer library version may
    include bits that were undefined at the time of
    compilation.
'''
        return libnbdmod.get_handshake_flags(self._o)

    def set_strict_mode(self, flags):
        '''▶ control how strictly to follow NBD protocol

    By default, libnbd tries to detect requests that would
    trigger undefined behavior in the NBD protocol, and
    rejects them client side without causing any network
    traffic, rather than risking undefined server behavior.
    However, for integration testing, it can be handy to
    relax the strictness of libnbd, to coerce it into
    sending such requests over the network for testing the
    robustness of the server in dealing with such traffic.

    The "flags" argument is a bitmask, including zero or
    more of the following strictness flags:

    "STRICT_COMMANDS" = 1
        If set, this flag rejects client requests that do
        not comply with the set of advertised server flags
        (for example, attempting a write on a read-only
        server, or attempting to use "CMD_FLAG_FUA" when
        "nbd.can_fua" returned false). If clear, this flag
        relies on the server to reject unexpected commands.

    "STRICT_FLAGS" = 2
        If set, this flag rejects client requests that
        attempt to set a command flag not recognized by
        libnbd (those outside of "CMD_FLAG_MASK"), or a flag
        not normally associated with a command (such as
        using "CMD_FLAG_FUA" on a read command). If clear,
        all flags are sent on to the server, even if sending
        such a flag may cause the server to change its reply
        in a manner that confuses libnbd, perhaps causing
        deadlock or ending the connection.

        Flags that are known by libnbd as associated with a
        given command (such as "CMD_FLAG_DF" for
        "nbd.pread_structured" gated by "nbd.can_df") are
        controlled by "STRICT_COMMANDS" instead.

        Note that the NBD protocol only supports 16 bits of
        command flags, even though the libnbd API uses
        "uint32_t"; bits outside of the range permitted by
        the protocol are always a client-side error.

    "STRICT_BOUNDS" = 3
        If set, this flag rejects client requests that would
        exceed the export bounds without sending any traffic
        to the server. If clear, this flag relies on the
        server to detect out-of-bounds requests.

    "STRICT_ZERO_SIZE" = 4
        If set, this flag rejects client requests with
        length 0. If clear, this permits zero-length
        requests to the server, which may produce undefined
        results.

    "STRICT_ALIGN" = 5
        If set, and the server provided minimum block sizes
        (see "nbd.get_block_size", this flag rejects client
        requests that do not have length and offset aligned
        to the server's minimum requirements. If clear,
        unaligned requests are sent to the server, where it
        is up to the server whether to honor or reject the
        request.

    For convenience, the constant "STRICT_MASK" is available
    to describe all strictness flags supported by this build
    of libnbd. Future versions of libnbd may add further
    flags, which are likely to be enabled by default for
    additional client-side filtering. As such, when
    attempting to relax only one specific bit while keeping
    remaining checks at the client side, it is wiser to
    first call "nbd.get_strict_mode" and modify that value,
    rather than blindly setting a constant value.
'''
        return libnbdmod.set_strict_mode(self._o, flags)

    def get_strict_mode(self):
        '''▶ see which strictness flags are in effect

    Return flags indicating which protocol strictness items
    are being enforced locally by libnbd rather than the
    server. The return value from a newer library version
    may include bits that were undefined at the time of
    compilation.
'''
        return libnbdmod.get_strict_mode(self._o)

    def set_opt_mode(self, enable):
        '''▶ control option mode, for pausing during option negotiation

    Set this flag to true in order to request that a
    connection command "nbd_connect_*" will pause for
    negotiation options rather than proceeding all the way
    to the ready state, when communicating with a newstyle
    server. This setting has no effect when connecting to an
    oldstyle server.

    When option mode is enabled, you have fine-grained
    control over which options are negotiated, compared to
    the default of the server negotiating everything on your
    behalf using settings made before starting the
    connection. To leave the mode and proceed on to the
    ready state, you must use "nbd.opt_go" successfully; a
    failed "nbd.opt_go" returns to the negotiating state to
    allow a change of export name before trying again. You
    may also use "nbd.opt_abort" to end the connection
    without finishing negotiation.
'''
        return libnbdmod.set_opt_mode(self._o, enable)

    def get_opt_mode(self):
        '''▶ return whether option mode was enabled

    Return true if option negotiation mode was enabled on
    this handle.
'''
        return libnbdmod.get_opt_mode(self._o)

    def opt_go(self):
        '''▶ end negotiation and move on to using an export

    Request that the server finish negotiation and move on
    to serving the export previously specified by the most
    recent "nbd.set_export_name" or "nbd.connect_uri". This
    can only be used if "nbd.set_opt_mode" enabled option
    mode.

    If this fails, the server may still be in negotiation,
    where it is possible to attempt another option such as a
    different export name; although older servers will
    instead have killed the connection.
'''
        return libnbdmod.opt_go(self._o)

    def opt_abort(self):
        '''▶ end negotiation and close the connection

    Request that the server finish negotiation, gracefully
    if possible, then close the connection. This can only be
    used if "nbd.set_opt_mode" enabled option mode.
'''
        return libnbdmod.opt_abort(self._o)

    def opt_list(self, list):
        '''▶ request the server to list all exports during negotiation

    Request that the server list all exports that it
    supports. This can only be used if "nbd.set_opt_mode"
    enabled option mode.

    The "list" function is called once per advertised
    export, with any "user_data" passed to this function,
    and with "name" and "description" supplied by the
    server. Many servers omit descriptions, in which case
    "description" will be an empty string. Remember that it
    is not safe to call "nbd.set_export_name" from within
    the context of the callback function; rather, your code
    must copy any "name" needed for later use after this
    function completes. At present, the return value of the
    callback is ignored, although a return of -1 should be
    avoided.

    For convenience, when this function succeeds, it returns
    the number of exports that were advertised by the
    server.

    Not all servers understand this request, and even when
    it is understood, the server might intentionally send an
    empty list to avoid being an information leak, may
    encounter a failure after delivering partial results, or
    may refuse to answer more than one query per connection
    in the interest of avoiding negotiation that does not
    resolve. Thus, this function may succeed even when no
    exports are reported, or may fail but have a non-empty
    list. Likewise, the NBD protocol does not specify an
    upper bound for the number of exports that might be
    advertised, so client code should be aware that a server
    may send a lengthy list.

    For nbd-server(1) you will need to allow clients to make
    list requests by adding "allowlist=true" to the
    "[generic]" section of /etc/nbd-server/config. For
    qemu-nbd(8), a description is set with *-D*.
'''
        return libnbdmod.opt_list(self._o, list)

    def opt_info(self):
        '''▶ request the server for information about an export

    Request that the server supply information about the
    export name previously specified by the most recent
    "nbd.set_export_name" or "nbd.connect_uri". This can
    only be used if "nbd.set_opt_mode" enabled option mode.

    If successful, functions like "nbd.is_read_only" and
    "nbd.get_size" will report details about that export. In
    general, if "nbd.opt_go" is called next, that call will
    likely succeed with the details remaining the same,
    although this is not guaranteed by all servers.

    Not all servers understand this request, and even when
    it is understood, the server might fail the request even
    when a corresponding "nbd.opt_go" would succeed.
'''
        return libnbdmod.opt_info(self._o)

    def opt_list_meta_context(self, context):
        '''▶ request the server to list available meta contexts

    Request that the server list available meta contexts
    associated with the export previously specified by the
    most recent "nbd.set_export_name" or "nbd.connect_uri".
    This can only be used if "nbd.set_opt_mode" enabled
    option mode.

    The NBD protocol allows a client to decide how many
    queries to ask the server. Rather than taking that list
    of queries as a parameter to this function, libnbd
    reuses the current list of requested meta contexts as
    set by "nbd.add_meta_context"; you can use
    "nbd.clear_meta_contexts" to set up a different list of
    queries. When the list is empty, a server will typically
    reply with all contexts that it supports; when the list
    is non-empty, the server will reply only with supported
    contexts that match the client's request. Note that a
    reply by the server might be encoded to represent
    several feasible contexts within one string, rather than
    multiple strings per actual context name that would
    actually succeed during "nbd.opt_go"; so it is still
    necessary to use "nbd.can_meta_context" after connecting
    to see which contexts are actually supported.

    The "context" function is called once per server reply,
    with any "user_data" passed to this function, and with
    "name" supplied by the server. Remember that it is not
    safe to call "nbd.add_meta_context" from within the
    context of the callback function; rather, your code must
    copy any "name" needed for later use after this function
    completes. At present, the return value of the callback
    is ignored, although a return of -1 should be avoided.

    For convenience, when this function succeeds, it returns
    the number of replies returned by the server.

    Not all servers understand this request, and even when
    it is understood, the server might intentionally send an
    empty list because it does not support the requested
    context, or may encounter a failure after delivering
    partial results. Thus, this function may succeed even
    when no contexts are reported, or may fail but have a
    non-empty list. Likewise, the NBD protocol does not
    specify an upper bound for the number of replies that
    might be advertised, so client code should be aware that
    a server may send a lengthy list.
'''
        return libnbdmod.opt_list_meta_context(self._o, context)

    def add_meta_context(self, name):
        '''▶ ask server to negotiate metadata context

    During connection libnbd can negotiate zero or more
    metadata contexts with the server. Metadata contexts are
    features (such as "base:allocation") which describe
    information returned by the "nbd.block_status" command
    (for "base:allocation" this is whether blocks of data
    are allocated, zero or sparse).

    This call adds one metadata context to the list to be
    negotiated. You can call it as many times as needed. The
    list is initially empty when the handle is created; you
    can check the contents of the list with
    "nbd.get_nr_meta_contexts" and "nbd.get_meta_context",
    or clear it with "nbd.clear_meta_contexts".

    The NBD protocol limits meta context names to 4096
    bytes, but servers may not support the full length. The
    encoding of meta context names is always UTF-8.

    Not all servers support all metadata contexts. To learn
    if a context was actually negotiated, call
    "nbd.can_meta_context" after connecting.

    The single parameter is the name of the metadata
    context, for example "CONTEXT_BASE_ALLOCATION".
    <libnbd.h> includes defined constants beginning with
    "CONTEXT_" for some well-known contexts, but you are
    free to pass in other contexts.

    Other metadata contexts are server-specific, but include
    "qemu:dirty-bitmap:..." and "qemu:allocation-depth" for
    qemu-nbd (see qemu-nbd *-B* and *-A* options).
'''
        return libnbdmod.add_meta_context(self._o, name)

    def get_nr_meta_contexts(self):
        '''▶ return the current number of requested meta contexts

    During connection libnbd can negotiate zero or more
    metadata contexts with the server. Metadata contexts are
    features (such as "base:allocation") which describe
    information returned by the "nbd.block_status" command
    (for "base:allocation" this is whether blocks of data
    are allocated, zero or sparse).

    This command returns how many meta contexts have been
    added to the list to request from the server via
    "nbd.add_meta_context". The server is not obligated to
    honor all of the requests; to see what it actually
    supports, see "nbd.can_meta_context".
'''
        return libnbdmod.get_nr_meta_contexts(self._o)

    def get_meta_context(self, i):
        '''▶ return the i'th meta context request

    During connection libnbd can negotiate zero or more
    metadata contexts with the server. Metadata contexts are
    features (such as "base:allocation") which describe
    information returned by the "nbd.block_status" command
    (for "base:allocation" this is whether blocks of data
    are allocated, zero or sparse).

    This command returns the i'th meta context request, as
    added by "nbd.add_meta_context", and bounded by
    "nbd.get_nr_meta_contexts".
'''
        return libnbdmod.get_meta_context(self._o, i)

    def clear_meta_contexts(self):
        '''▶ reset the list of requested meta contexts

    During connection libnbd can negotiate zero or more
    metadata contexts with the server. Metadata contexts are
    features (such as "base:allocation") which describe
    information returned by the "nbd.block_status" command
    (for "base:allocation" this is whether blocks of data
    are allocated, zero or sparse).

    This command resets the list of meta contexts to request
    back to an empty list, for re-population by further use
    of "nbd.add_meta_context". It is primarily useful when
    option negotiation mode is selected (see
    "nbd.set_opt_mode"), for altering the list of attempted
    contexts between subsequent export queries.
'''
        return libnbdmod.clear_meta_contexts(self._o)

    def set_uri_allow_transports(self, mask):
        '''▶ set the allowed transports in NBD URIs

    Set which transports are allowed to appear in NBD URIs.
    The default is to allow any transport.

    The "mask" parameter may contain any of the following
    flags ORed together:

    "ALLOW_TRANSPORT_TCP"
    "ALLOW_TRANSPORT_UNIX"
    "ALLOW_TRANSPORT_VSOCK"

    For convenience, the constant "ALLOW_TRANSPORT_MASK" is
    available to describe all transports recognized by this
    build of libnbd. A future version of the library may add
    new flags.
'''
        return libnbdmod.set_uri_allow_transports(self._o, mask)

    def set_uri_allow_tls(self, tls):
        '''▶ set the allowed TLS settings in NBD URIs

    Set which TLS settings are allowed to appear in NBD
    URIs. The default is to allow either non-TLS or TLS
    URIs.

    The "tls" parameter can be:

    "TLS_DISABLE"
        TLS URIs are not permitted, ie. a URI such as
        "nbds://..." will be rejected.

    "TLS_ALLOW"
        This is the default. TLS may be used or not,
        depending on whether the URI uses "nbds" or "nbd".

    "TLS_REQUIRE"
        TLS URIs are required. All URIs must use "nbds".
'''
        return libnbdmod.set_uri_allow_tls(self._o, tls)

    def set_uri_allow_local_file(self, allow):
        '''▶ set the allowed transports in NBD URIs

    Allow NBD URIs to reference local files. This is
    *disabled* by default.

    Currently this setting only controls whether the
    "tls-psk-file" parameter in NBD URIs is allowed.
'''
        return libnbdmod.set_uri_allow_local_file(self._o, allow)

    def connect_uri(self, uri):
        '''▶ connect to NBD URI

    Connect (synchronously) to an NBD server and export by
    specifying the NBD URI. This call parses the URI and
    calls "nbd.set_export_name" and "nbd.set_tls" and other
    calls as needed, followed by "nbd.connect_tcp" or
    "nbd.connect_unix". However, it is possible to override
    the export name portion of a URI by using
    "nbd.set_opt_mode" to enable option mode, then using
    "nbd.set_export_name" and "nbd.opt_go" as part of
    subsequent negotiation.

    This call returns when the connection has been made.

  Example URIs supported
    "nbd://example.com"
        Connect over TCP, unencrypted, to "example.com" port
        10809.

    "nbds://example.com"
        Connect over TCP with TLS, to "example.com" port
        10809. If the server does not support TLS then this
        will fail.

    "nbd+unix:///foo?socket=/tmp/nbd.sock"
        Connect over the Unix domain socket /tmp/nbd.sock to
        an NBD server running locally. The export name is
        set to "foo" (note without any leading "/"
        character).

    "nbd+vsock:///"
        In this scenario libnbd is running in a virtual
        machine. Connect over "AF_VSOCK" to an NBD server
        running on the hypervisor.

  Supported URI formats
    The following schemes are supported in the current
    version of libnbd:

    "nbd:"
        Connect over TCP without using TLS.

    "nbds:"
        Connect over TCP. TLS is required and the connection
        will fail if the server does not support TLS.

    "nbd+unix:"
    "nbds+unix:"
        Connect over a Unix domain socket, without or with
        TLS respectively. The "socket" parameter is
        required.

    "nbd+vsock:"
    "nbds+vsock:"
        Connect over the "AF_VSOCK" transport, without or
        with TLS respectively.

    The authority part of the URI
    ("[username@][servername][:port]") is parsed depending
    on the transport. For TCP it specifies the server to
    connect to and optional port number. For "+unix" it
    should not be present. For "+vsock" the server name is
    the numeric CID (eg. 2 to connect to the host), and the
    optional port number may be present. If the "username"
    is present it is used for TLS authentication.

    For all transports, an export name may be present,
    parsed in accordance with the NBD URI specification.

    Finally the query part of the URI can contain:

    socket=SOCKET
        Specifies the Unix domain socket to connect on. Must
        be present for the "+unix" transport and must not be
        present for the other transports.

    tls-psk-file=PSKFILE
        Set the PSK file. See "nbd.set_tls_psk_file". Note
        this is not allowed by default - see next section.

  Disable URI features
    For security reasons you might want to disable certain
    URI features. Pre-filtering URIs is error-prone and
    should not be attempted. Instead use the libnbd APIs
    below to control what can appear in URIs. Note you must
    call these functions on the same handle before calling
    "nbd.connect_uri" or "nbd.aio_connect_uri".

    TCP, Unix domain socket or "AF_VSOCK" transports
        Default: all allowed

        To select which transports are allowed call
        "nbd.set_uri_allow_transports".

    TLS Default: both non-TLS and TLS connections allowed

        To force TLS off or on in URIs call
        "nbd.set_uri_allow_tls".

    Connect to Unix domain socket in the local filesystem
        Default: allowed

        To prevent this you must disable the "+unix"
        transport using "nbd.set_uri_allow_transports".

    Read from local files
        Default: denied

        To allow URIs to contain references to local files
        (eg. for parameters like "tls-psk-file") call
        "nbd.set_uri_allow_local_file".

  Optional features
    This call will fail if libnbd was not compiled with
    libxml2; you can test whether this is the case with
    "nbd.supports_uri".

    Support for URIs that require TLS will fail if libnbd
    was not compiled with gnutls; you can test whether this
    is the case with "nbd.supports_tls".

  Constructing a URI from an existing connection
    See "nbd.get_uri".
'''
        return libnbdmod.connect_uri(self._o, uri)

    def connect_unix(self, unixsocket):
        '''▶ connect to NBD server over a Unix domain socket

    Connect (synchronously) over the named Unix domain
    socket ("unixsocket") to an NBD server running on the
    same machine. This call returns when the connection has
    been made.
'''
        return libnbdmod.connect_unix(self._o, unixsocket)

    def connect_vsock(self, cid, port):
        '''▶ connect to NBD server over AF_VSOCK protocol

    Connect (synchronously) over the "AF_VSOCK" protocol
    from a virtual machine to an NBD server, usually running
    on the host. The "cid" and "port" parameters specify the
    server address. Usually "cid" should be 2 (to connect to
    the host), and "port" might be 10809 or another port
    number assigned to you by the host administrator. This
    call returns when the connection has been made.
'''
        return libnbdmod.connect_vsock(self._o, cid, port)

    def connect_tcp(self, hostname, port):
        '''▶ connect to NBD server over a TCP port

    Connect (synchronously) to the NBD server listening on
    "hostname:port". The "port" may be a port name such as
    "nbd", or it may be a port number as a string such as
    "10809". This call returns when the connection has been
    made.
'''
        return libnbdmod.connect_tcp(self._o, hostname, port)

    def connect_socket(self, sock):
        '''▶ connect directly to a connected socket

    Pass a connected socket "sock" through which libnbd will
    talk to the NBD server.

    The caller is responsible for creating and connecting
    this socket by some method, before passing it to libnbd.

    If this call returns without error then socket ownership
    is passed to libnbd. Libnbd will close the socket when
    the handle is closed. The caller must not use the socket
    in any way.
'''
        return libnbdmod.connect_socket(self._o, sock)

    def connect_command(self, argv):
        '''▶ connect to NBD server command

    Run the command as a subprocess and connect to it over
    stdin/stdout. This is for use with NBD servers which can
    behave like inetd clients, such as "nbdkit --single".

  Subprocess
    Libnbd will fork the "argv" command and pass the NBD
    socket to it using file descriptors 0 and 1
    (stdin/stdout):

     ┌─────────┬─────────┐    ┌────────────────┐
     │ program │ libnbd  │    │   NBD server   │
     │         │         │    │       (argv)   │
     │         │ socket ╍╍╍╍╍╍╍╍▶ stdin/stdout │
     └─────────┴─────────┘    └────────────────┘

    When the NBD handle is closed the server subprocess is
    killed.
'''
        return libnbdmod.connect_command(self._o, argv)

    def connect_systemd_socket_activation(self, argv):
        '''▶ connect using systemd socket activation

    Run the command as a subprocess and connect to it using
    systemd socket activation.

    This is especially useful for running qemu-nbd(1) as a
    subprocess of libnbd, for example to use it to open
    qcow2 files. To run nbdkit as a subprocess it is usually
    better to use "nbd.connect_command".

  Socket activation
    Libnbd will fork the "argv" command and pass an NBD
    socket to it using special "LISTEN_*" environment
    variables (as defined by the systemd socket activation
    protocol).

     ┌─────────┬─────────┐    ┌───────────────┐
     │ program │ libnbd  │    │  qemu-nbd or  │
     │         │         │    │  other server │
     │         │ socket ╍╍╍╍╍╍╍╍▶             │
     └─────────┴─────────┘    └───────────────┘

    When the NBD handle is closed the server subprocess is
    killed.
'''
        return libnbdmod.connect_systemd_socket_activation(self._o, argv)

    def is_read_only(self):
        '''▶ is the NBD export read-only?

    Returns true if the NBD export is read-only; writes and
    write-like operations will fail.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.is_read_only(self._o)

    def can_flush(self):
        '''▶ does the server support the flush command?

    Returns true if the server supports the flush command
    (see "nbd.flush", "nbd.aio_flush"). Returns false if the
    server does not.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.can_flush(self._o)

    def can_fua(self):
        '''▶ does the server support the FUA flag?

    Returns true if the server supports the FUA flag on
    certain commands (see "nbd.pwrite").

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.can_fua(self._o)

    def is_rotational(self):
        '''▶ is the NBD disk rotational (like a disk)?

    Returns true if the disk exposed over NBD is rotational
    (like a traditional floppy or hard disk). Returns false
    if the disk has no penalty for random access (like an
    SSD or RAM disk).

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.is_rotational(self._o)

    def can_trim(self):
        '''▶ does the server support the trim command?

    Returns true if the server supports the trim command
    (see "nbd.trim", "nbd.aio_trim"). Returns false if the
    server does not.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.can_trim(self._o)

    def can_zero(self):
        '''▶ does the server support the zero command?

    Returns true if the server supports the zero command
    (see "nbd.zero", "nbd.aio_zero"). Returns false if the
    server does not.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.can_zero(self._o)

    def can_fast_zero(self):
        '''▶ does the server support the fast zero flag?

    Returns true if the server supports the use of the
    "CMD_FLAG_FAST_ZERO" flag to the zero command (see
    "nbd.zero", "nbd.aio_zero"). Returns false if the server
    does not.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.can_fast_zero(self._o)

    def can_df(self):
        '''▶ does the server support the don't fragment flag to pread?

    Returns true if the server supports structured reads
    with an ability to request a non-fragmented read (see
    "nbd.pread_structured", "nbd.aio_pread_structured").
    Returns false if the server either lacks structured
    reads or if it does not support a non-fragmented read
    request.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.can_df(self._o)

    def can_multi_conn(self):
        '''▶ does the server support multi-conn?

    Returns true if the server supports multi-conn. Returns
    false if the server does not.

    It is not safe to open multiple handles connecting to
    the same server if you will write to the server and the
    server does not advertise multi-conn support. The safe
    way to check for this is to open one connection, check
    this flag is true, then open further connections as
    required.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.can_multi_conn(self._o)

    def can_cache(self):
        '''▶ does the server support the cache command?

    Returns true if the server supports the cache command
    (see "nbd.cache", "nbd.aio_cache"). Returns false if the
    server does not.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.can_cache(self._o)

    def can_meta_context(self, metacontext):
        '''▶ does the server support a specific meta context?

    Returns true if the server supports the given meta
    context (see "nbd.add_meta_context"). Returns false if
    the server does not.

    The single parameter is the name of the metadata
    context, for example "CONTEXT_BASE_ALLOCATION".
    <libnbd.h> includes defined constants for well-known
    namespace contexts beginning with "CONTEXT_", but you
    are free to pass in other contexts.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.can_meta_context(self._o, metacontext)

    def get_protocol(self):
        '''▶ return the NBD protocol variant

    Return the NBD protocol variant in use on the
    connection. At the moment this returns one of the
    strings "oldstyle", "newstyle" or "newstyle-fixed".
    Other strings might be returned in the future. Most
    modern NBD servers use "newstyle-fixed".

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.get_protocol(self._o)

    def get_size(self):
        '''▶ return the export size

    Returns the size in bytes of the NBD export.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.get_size(self._o)

    def get_block_size(self, size_type):
        '''▶ return a specific server block size constraint

    Returns a specific size constraint advertised by the
    server, if any. If the return is zero, the server did
    not advertise a constraint. "size_type" must be one of
    the following constraints:

    "SIZE_MINIMUM" = 0
        If non-zero, this will be a power of 2 between 1 and
        64k; any client request that is not aligned in
        length or offset to this size is likely to fail with
        "EINVAL". The image size will generally also be a
        multiple of this value (if not, the final few bytes
        are inaccessible while obeying alignment
        constraints). If zero, it is safest to assume a
        minimum block size of 512, although many servers
        support a minimum block size of 1. If the server
        provides a constraint, then libnbd defaults to
        honoring that constraint client-side unless
        "STRICT_ALIGN" is cleared in nbd_set_strict_mode(3).

    "SIZE_PREFERRED" = 1
        If non-zero, this is a power of 2 representing the
        preferred size for efficient I/O. Smaller requests
        may incur overhead such as read-modify-write cycles
        that will not be present when using I/O that is a
        multiple of this value. This value may be larger
        than the size of the export. If zero, using 4k as a
        preferred block size tends to give decent
        performance.

    "SIZE_MAXIMUM" = 2
        If non-zero, this represents the maximum length that
        the server is willing to handle during "nbd.pread"
        or "nbd.pwrite". Other functions like "nbd.zero" may
        still be able to use larger sizes. Note that this
        function returns what the server advertised, but
        libnbd itself imposes a maximum of 64M. If zero,
        some NBD servers will abruptly disconnect if a
        transaction involves more than 32M.

    Future NBD extensions may result in additional
    "size_type" values.

    This call does not block, because it returns data that
    is saved in the handle from the NBD protocol handshake.
'''
        return libnbdmod.get_block_size(self._o, size_type)

    def pread(self, count, offset, flags=0):
        '''▶ read from the NBD server

    Issue a read command to the NBD server for the range
    starting at "offset" and ending at "offset" + "count" -
    1. NBD can only read all or nothing using this call. The
    call returns when the data has been read fully into
    "buf" or there is an error. See also
    "nbd.pread_structured", if finer visibility is required
    into the server's replies, or if you want to use
    "CMD_FLAG_DF".

    The "flags" parameter must be 0 for now (it exists for
    future NBD protocol extensions).

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.pread(self._o, count, offset, flags)

    def pread_structured(self, count, offset, chunk, flags=0):
        '''▶ read from the NBD server

    Issue a read command to the NBD server for the range
    starting at "offset" and ending at "offset" + "count" -
    1. The server's response may be subdivided into chunks
    which may arrive out of order before reassembly into the
    original buffer; the "chunk" callback is used for
    notification after each chunk arrives, and may perform
    additional sanity checking on the server's reply. The
    callback cannot call "nbd_*" APIs on the same handle
    since it holds the handle lock and will cause a
    deadlock. If the callback returns -1, and no earlier
    error has been detected, then the overall read command
    will fail with any non-zero value stored into the
    callback's "error" parameter (with a default of
    "EPROTO"); but any further chunks will still invoke the
    callback.

    The "chunk" function is called once per chunk of data
    received, with the "user_data" passed to this function.
    The "subbuf" and "count" parameters represent the subset
    of the original buffer which has just been populated by
    results from the server (in C, "subbuf" always points
    within the original "buf"; but this guarantee may not
    extend to other language bindings). The "offset"
    parameter represents the absolute offset at which
    "subbuf" begins within the image (note that this is not
    the relative offset of "subbuf" within the original
    buffer "buf"). Changes to "error" on output are ignored
    unless the callback fails. The input meaning of the
    "error" parameter is controlled by the "status"
    parameter, which is one of

    "READ_DATA" = 1
        "subbuf" was populated with "count" bytes of data.
        On input, "error" contains the errno value of any
        earlier detected error, or zero.

    "READ_HOLE" = 2
        "subbuf" represents a hole, and contains "count" NUL
        bytes. On input, "error" contains the errno value of
        any earlier detected error, or zero.

    "READ_ERROR" = 3
        "count" is 0, so "subbuf" is unusable. On input,
        "error" contains the errno value reported by the
        server as occurring while reading that "offset",
        regardless if any earlier error has been detected.

    Future NBD extensions may permit other values for
    "status", but those will not be returned to a client
    that has not opted in to requesting such extensions. If
    the server is non-compliant, it is possible for the
    "chunk" function to be called more times than you expect
    or with "count" 0 for "READ_DATA" or "READ_HOLE". It is
    also possible that the "chunk" function is not called at
    all (in particular, "READ_ERROR" is used only when an
    error is associated with a particular offset, and not
    when the server reports a generic error), but you are
    guaranteed that the callback was called at least once if
    the overall read succeeds. Libnbd does not validate that
    the server obeyed the requirement that a read call must
    not have overlapping chunks and must not succeed without
    enough chunks to cover the entire request.

    The "flags" parameter may be 0 for no flags, or may
    contain "CMD_FLAG_DF" meaning that the server should not
    reply with more than one fragment (if that is supported
    - some servers cannot do this, see "nbd.can_df"). Libnbd
    does not validate that the server actually obeys the
    flag.

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.pread_structured(self._o, count, offset, chunk,
                                          flags)

    def pwrite(self, buf, offset, flags=0):
        '''▶ write to the NBD server

    Issue a write command to the NBD server, writing the
    data in "buf" to the range starting at "offset" and
    ending at "offset" + "count" - 1. NBD can only write all
    or nothing using this call. The call returns when the
    command has been acknowledged by the server, or there is
    an error. Note this will generally return an error if
    "nbd.is_read_only" is true.

    The "flags" parameter may be 0 for no flags, or may
    contain "CMD_FLAG_FUA" meaning that the server should
    not return until the data has been committed to
    permanent storage (if that is supported - some servers
    cannot do this, see "nbd.can_fua").

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.pwrite(self._o, buf, offset, flags)

    def shutdown(self, flags=0):
        '''▶ disconnect from the NBD server

    Issue the disconnect command to the NBD server. This is
    a nice way to tell the server we are going away, but
    from the client's point of view has no advantage over
    abruptly closing the connection (see "nbd.close").

    This function works whether or not the handle is ready
    for transmission of commands. If more fine-grained
    control is needed, see "nbd.aio_disconnect".

    The "flags" argument is a bitmask, including zero or
    more of the following shutdown flags:

    "SHUTDOWN_ABANDON_PENDING" = 0x10000
        If there are any pending requests which have not yet
        been sent to the server (see "nbd.aio_in_flight"),
        abandon them without sending them to the server,
        rather than the usual practice of issuing those
        commands before informing the server of the intent
        to disconnect.

    For convenience, the constant "SHUTDOWN_MASK" is
    available to describe all shutdown flags recognized by
    this build of libnbd. A future version of the library
    may add new flags.
'''
        return libnbdmod.shutdown(self._o, flags)

    def flush(self, flags=0):
        '''▶ send flush command to the NBD server

    Issue the flush command to the NBD server. The function
    should return when all write commands which have
    completed have been committed to permanent storage on
    the server. Note this will generally return an error if
    "nbd.can_flush" is false.

    The "flags" parameter must be 0 for now (it exists for
    future NBD protocol extensions).

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.flush(self._o, flags)

    def trim(self, count, offset, flags=0):
        '''▶ send trim command to the NBD server

    Issue a trim command to the NBD server, which if
    supported by the server causes a hole to be punched in
    the backing store starting at "offset" and ending at
    "offset" + "count" - 1. The call returns when the
    command has been acknowledged by the server, or there is
    an error. Note this will generally return an error if
    "nbd.can_trim" is false or "nbd.is_read_only" is true.

    The "flags" parameter may be 0 for no flags, or may
    contain "CMD_FLAG_FUA" meaning that the server should
    not return until the data has been committed to
    permanent storage (if that is supported - some servers
    cannot do this, see "nbd.can_fua").

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.trim(self._o, count, offset, flags)

    def cache(self, count, offset, flags=0):
        '''▶ send cache (prefetch) command to the NBD server

    Issue the cache (prefetch) command to the NBD server,
    which if supported by the server causes data to be
    prefetched into faster storage by the server, speeding
    up a subsequent "nbd.pread" call. The server can also
    silently ignore this command. Note this will generally
    return an error if "nbd.can_cache" is false.

    The "flags" parameter must be 0 for now (it exists for
    future NBD protocol extensions).

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.cache(self._o, count, offset, flags)

    def zero(self, count, offset, flags=0):
        '''▶ send write zeroes command to the NBD server

    Issue a write zeroes command to the NBD server, which if
    supported by the server causes a zeroes to be written
    efficiently starting at "offset" and ending at "offset"
    + "count" - 1. The call returns when the command has
    been acknowledged by the server, or there is an error.
    Note this will generally return an error if
    "nbd.can_zero" is false or "nbd.is_read_only" is true.

    The "flags" parameter may be 0 for no flags, or may
    contain "CMD_FLAG_FUA" meaning that the server should
    not return until the data has been committed to
    permanent storage (if that is supported - some servers
    cannot do this, see "nbd.can_fua"), "CMD_FLAG_NO_HOLE"
    meaning that the server should favor writing actual
    allocated zeroes over punching a hole, and/or
    "CMD_FLAG_FAST_ZERO" meaning that the server must fail
    quickly if writing zeroes is no faster than a normal
    write (if that is supported - some servers cannot do
    this, see "nbd.can_fast_zero").

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.zero(self._o, count, offset, flags)

    def block_status(self, count, offset, extent, flags=0):
        '''▶ send block status command to the NBD server

    Issue the block status command to the NBD server. If
    supported by the server, this causes metadata context
    information about blocks beginning from the specified
    offset to be returned. The "count" parameter is a hint:
    the server may choose to return less status, or the
    final block may extend beyond the requested range. If
    multiple contexts are supported, the number of blocks
    and cumulative length of those blocks need not be
    identical between contexts.

    Depending on which metadata contexts were enabled before
    connecting (see "nbd.add_meta_context") and which are
    supported by the server (see "nbd.can_meta_context")
    this call returns information about extents by calling
    back to the "extent" function. The callback cannot call
    "nbd_*" APIs on the same handle since it holds the
    handle lock and will cause a deadlock. If the callback
    returns -1, and no earlier error has been detected, then
    the overall block status command will fail with any
    non-zero value stored into the callback's "error"
    parameter (with a default of "EPROTO"); but any further
    contexts will still invoke the callback.

    The "extent" function is called once per type of
    metadata available, with the "user_data" passed to this
    function. The "metacontext" parameter is a string such
    as "base:allocation". The "entries" array is an array of
    pairs of integers with the first entry in each pair
    being the length (in bytes) of the block and the second
    entry being a status/flags field which is specific to
    the metadata context. (The number of pairs passed to the
    function is "nr_entries/2".) The NBD protocol document
    in the section about "NBD_REPLY_TYPE_BLOCK_STATUS"
    describes the meaning of this array; for contexts known
    to libnbd, <libnbd.h> contains constants beginning with
    "STATE_" that may help decipher the values. On entry to
    the callback, the "error" parameter contains the errno
    value of any previously detected error.

    It is possible for the extent function to be called more
    times than you expect (if the server is buggy), so
    always check the "metacontext" field to ensure you are
    receiving the data you expect. It is also possible that
    the extent function is not called at all, even for
    metadata contexts that you requested. This indicates
    either that the server doesn't support the context or
    for some other reason cannot return the data.

    The "flags" parameter may be 0 for no flags, or may
    contain "CMD_FLAG_REQ_ONE" meaning that the server
    should return only one extent per metadata context where
    that extent does not exceed "count" bytes; however,
    libnbd does not validate that the server obeyed the
    flag.

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.block_status(self._o, count, offset, extent, flags)

    def poll(self, timeout):
        '''▶ poll the handle once

    This is a simple implementation of poll(2) which is used
    internally by synchronous API calls. On success, it
    returns 0 if the "timeout" (in milliseconds) occurs, or
    1 if the poll completed and the state machine
    progressed. Set "timeout" to -1 to block indefinitely
    (but be careful that eventual action is actually
    expected - for example, if the connection is established
    but there are no commands in flight, using an infinite
    timeout will permanently block).

    This function is mainly useful as an example of how you
    might integrate libnbd with your own main loop, rather
    than being intended as something you would use.
'''
        return libnbdmod.poll(self._o, timeout)

    def aio_connect(self, addr):
        '''▶ connect to the NBD server

    Begin connecting to the NBD server. The "addr" and
    "addrlen" parameters specify the address of the socket
    to connect to.

    You can check if the connection is still connecting by
    calling "nbd.aio_is_connecting", or if it has connected
    to the server and completed the NBD handshake by calling
    "nbd.aio_is_ready", on the connection.
'''
        return libnbdmod.aio_connect(self._o, addr)

    def aio_connect_uri(self, uri):
        '''▶ connect to an NBD URI

    Begin connecting to the NBD URI "uri". Parameters behave
    as documented in "nbd.connect_uri".

    You can check if the connection is still connecting by
    calling "nbd.aio_is_connecting", or if it has connected
    to the server and completed the NBD handshake by calling
    "nbd.aio_is_ready", on the connection.
'''
        return libnbdmod.aio_connect_uri(self._o, uri)

    def aio_connect_unix(self, unixsocket):
        '''▶ connect to the NBD server over a Unix domain socket

    Begin connecting to the NBD server over Unix domain
    socket ("unixsocket"). Parameters behave as documented
    in "nbd.connect_unix".

    You can check if the connection is still connecting by
    calling "nbd.aio_is_connecting", or if it has connected
    to the server and completed the NBD handshake by calling
    "nbd.aio_is_ready", on the connection.
'''
        return libnbdmod.aio_connect_unix(self._o, unixsocket)

    def aio_connect_vsock(self, cid, port):
        '''▶ connect to the NBD server over AF_VSOCK socket

    Begin connecting to the NBD server over the "AF_VSOCK"
    protocol to the server "cid:port". Parameters behave as
    documented in "nbd.connect_vsock".

    You can check if the connection is still connecting by
    calling "nbd.aio_is_connecting", or if it has connected
    to the server and completed the NBD handshake by calling
    "nbd.aio_is_ready", on the connection.
'''
        return libnbdmod.aio_connect_vsock(self._o, cid, port)

    def aio_connect_tcp(self, hostname, port):
        '''▶ connect to the NBD server over a TCP port

    Begin connecting to the NBD server listening on
    "hostname:port". Parameters behave as documented in
    "nbd.connect_tcp".

    You can check if the connection is still connecting by
    calling "nbd.aio_is_connecting", or if it has connected
    to the server and completed the NBD handshake by calling
    "nbd.aio_is_ready", on the connection.
'''
        return libnbdmod.aio_connect_tcp(self._o, hostname, port)

    def aio_connect_socket(self, sock):
        '''▶ connect directly to a connected socket

    Begin connecting to the connected socket "fd".
    Parameters behave as documented in "nbd.connect_socket".

    You can check if the connection is still connecting by
    calling "nbd.aio_is_connecting", or if it has connected
    to the server and completed the NBD handshake by calling
    "nbd.aio_is_ready", on the connection.
'''
        return libnbdmod.aio_connect_socket(self._o, sock)

    def aio_connect_command(self, argv):
        '''▶ connect to the NBD server

    Run the command as a subprocess and begin connecting to
    it over stdin/stdout. Parameters behave as documented in
    "nbd.connect_command".

    You can check if the connection is still connecting by
    calling "nbd.aio_is_connecting", or if it has connected
    to the server and completed the NBD handshake by calling
    "nbd.aio_is_ready", on the connection.
'''
        return libnbdmod.aio_connect_command(self._o, argv)

    def aio_connect_systemd_socket_activation(self, argv):
        '''▶ connect using systemd socket activation

    Run the command as a subprocess and begin connecting to
    it using systemd socket activation. Parameters behave as
    documented in "nbd.connect_systemd_socket_activation".

    You can check if the connection is still connecting by
    calling "nbd.aio_is_connecting", or if it has connected
    to the server and completed the NBD handshake by calling
    "nbd.aio_is_ready", on the connection.
'''
        return libnbdmod.aio_connect_systemd_socket_activation(self._o,
                                                               argv)

    def aio_opt_go(self, completion=None):
        '''▶ end negotiation and move on to using an export

    Request that the server finish negotiation and move on
    to serving the export previously specified by the most
    recent "nbd.set_export_name" or "nbd.connect_uri". This
    can only be used if "nbd.set_opt_mode" enabled option
    mode.

    To determine when the request completes, wait for
    "nbd.aio_is_connecting" to return false. Or supply the
    optional "completion_callback" which will be invoked as
    described in "Completion callbacks" in libnbd(3), except
    that it is automatically retired regardless of return
    value. Note that directly detecting whether the server
    returns an error (as is done by the return value of the
    synchronous counterpart) is only possible with a
    completion callback; however it is also possible to
    indirectly detect an error when "nbd.aio_is_negotiating"
    returns true.
'''
        return libnbdmod.aio_opt_go(self._o, completion)

    def aio_opt_abort(self):
        '''▶ end negotiation and close the connection

    Request that the server finish negotiation, gracefully
    if possible, then close the connection. This can only be
    used if "nbd.set_opt_mode" enabled option mode.

    To determine when the request completes, wait for
    "nbd.aio_is_connecting" to return false.
'''
        return libnbdmod.aio_opt_abort(self._o)

    def aio_opt_list(self, list, completion=None):
        '''▶ request the server to list all exports during negotiation

    Request that the server list all exports that it
    supports. This can only be used if "nbd.set_opt_mode"
    enabled option mode.

    To determine when the request completes, wait for
    "nbd.aio_is_connecting" to return false. Or supply the
    optional "completion_callback" which will be invoked as
    described in "Completion callbacks" in libnbd(3), except
    that it is automatically retired regardless of return
    value. Note that detecting whether the server returns an
    error (as is done by the return value of the synchronous
    counterpart) is only possible with a completion
    callback.
'''
        return libnbdmod.aio_opt_list(self._o, list, completion)

    def aio_opt_info(self, completion=None):
        '''▶ request the server for information about an export

    Request that the server supply information about the
    export name previously specified by the most recent
    "nbd.set_export_name" or "nbd.connect_uri". This can
    only be used if "nbd.set_opt_mode" enabled option mode.

    To determine when the request completes, wait for
    "nbd.aio_is_connecting" to return false. Or supply the
    optional "completion_callback" which will be invoked as
    described in "Completion callbacks" in libnbd(3), except
    that it is automatically retired regardless of return
    value. Note that detecting whether the server returns an
    error (as is done by the return value of the synchronous
    counterpart) is only possible with a completion
    callback.
'''
        return libnbdmod.aio_opt_info(self._o, completion)

    def aio_opt_list_meta_context(self, context, completion=None):
        '''▶ request the server to list available meta contexts

    Request that the server list available meta contexts
    associated with the export previously specified by the
    most recent "nbd.set_export_name" or "nbd.connect_uri".
    This can only be used if "nbd.set_opt_mode" enabled
    option mode.

    To determine when the request completes, wait for
    "nbd.aio_is_connecting" to return false. Or supply the
    optional "completion_callback" which will be invoked as
    described in "Completion callbacks" in libnbd(3), except
    that it is automatically retired regardless of return
    value. Note that detecting whether the server returns an
    error (as is done by the return value of the synchronous
    counterpart) is only possible with a completion
    callback.
'''
        return libnbdmod.aio_opt_list_meta_context(self._o, context,
                                                   completion)

    def aio_pread(self, buf, offset, completion=None, flags=0):
        '''▶ read from the NBD server

    Issue a read command to the NBD server.

    To check if the command completed, call
    "nbd.aio_command_completed". Or supply the optional
    "completion_callback" which will be invoked as described
    in "Completion callbacks" in libnbd(3).

    Note that you must ensure "buf" is valid until the
    command has completed. Other parameters behave as
    documented in "nbd.pread".

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.aio_pread(self._o, buf._o, offset, completion,
                                   flags)

    def aio_pread_structured(self, buf, offset, chunk, completion=None,
                             flags=0):
        '''▶ read from the NBD server

    Issue a read command to the NBD server.

    To check if the command completed, call
    "nbd.aio_command_completed". Or supply the optional
    "completion_callback" which will be invoked as described
    in "Completion callbacks" in libnbd(3).

    Other parameters behave as documented in
    "nbd.pread_structured".

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.aio_pread_structured(self._o, buf._o, offset,
                                              chunk, completion, flags)

    def aio_pwrite(self, buf, offset, completion=None, flags=0):
        '''▶ write to the NBD server

    Issue a write command to the NBD server.

    To check if the command completed, call
    "nbd.aio_command_completed". Or supply the optional
    "completion_callback" which will be invoked as described
    in "Completion callbacks" in libnbd(3).

    Note that you must ensure "buf" is valid until the
    command has completed. Other parameters behave as
    documented in "nbd.pwrite".

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.aio_pwrite(self._o, buf._o, offset, completion,
                                    flags)

    def aio_disconnect(self, flags=0):
        '''▶ disconnect from the NBD server

    Issue the disconnect command to the NBD server. This is
    not a normal command because NBD servers are not obliged
    to send a reply. Instead you should wait for
    "nbd.aio_is_closed" to become true on the connection.
    Once this command is issued, you cannot issue any
    further commands.

    Although libnbd does not prevent you from issuing this
    command while still waiting on the replies to previous
    commands, the NBD protocol recommends that you wait
    until there are no other commands in flight (see
    "nbd.aio_in_flight"), to give the server a better chance
    at a clean shutdown.

    The "flags" parameter must be 0 for now (it exists for
    future NBD protocol extensions). There is no direct
    synchronous counterpart; however, "nbd.shutdown" will
    call this function if appropriate.
'''
        return libnbdmod.aio_disconnect(self._o, flags)

    def aio_flush(self, completion=None, flags=0):
        '''▶ send flush command to the NBD server

    Issue the flush command to the NBD server.

    To check if the command completed, call
    "nbd.aio_command_completed". Or supply the optional
    "completion_callback" which will be invoked as described
    in "Completion callbacks" in libnbd(3).

    Other parameters behave as documented in "nbd.flush".

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.aio_flush(self._o, completion, flags)

    def aio_trim(self, count, offset, completion=None, flags=0):
        '''▶ send trim command to the NBD server

    Issue a trim command to the NBD server.

    To check if the command completed, call
    "nbd.aio_command_completed". Or supply the optional
    "completion_callback" which will be invoked as described
    in "Completion callbacks" in libnbd(3).

    Other parameters behave as documented in "nbd.trim".

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.aio_trim(self._o, count, offset, completion, flags)

    def aio_cache(self, count, offset, completion=None, flags=0):
        '''▶ send cache (prefetch) command to the NBD server

    Issue the cache (prefetch) command to the NBD server.

    To check if the command completed, call
    "nbd.aio_command_completed". Or supply the optional
    "completion_callback" which will be invoked as described
    in "Completion callbacks" in libnbd(3).

    Other parameters behave as documented in "nbd.cache".

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.aio_cache(self._o, count, offset, completion,
                                   flags)

    def aio_zero(self, count, offset, completion=None, flags=0):
        '''▶ send write zeroes command to the NBD server

    Issue a write zeroes command to the NBD server.

    To check if the command completed, call
    "nbd.aio_command_completed". Or supply the optional
    "completion_callback" which will be invoked as described
    in "Completion callbacks" in libnbd(3).

    Other parameters behave as documented in "nbd.zero".

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.aio_zero(self._o, count, offset, completion, flags)

    def aio_block_status(self, count, offset, extent, completion=None,
                         flags=0):
        '''▶ send block status command to the NBD server

    Send the block status command to the NBD server.

    To check if the command completed, call
    "nbd.aio_command_completed". Or supply the optional
    "completion_callback" which will be invoked as described
    in "Completion callbacks" in libnbd(3).

    Other parameters behave as documented in
    "nbd.block_status".

    By default, libnbd will reject attempts to use this
    function with parameters that are likely to result in
    server failure, such as requesting an unknown command
    flag. The "nbd.set_strict_mode" function can be used to
    alter which scenarios should await a server reply rather
    than failing fast.
'''
        return libnbdmod.aio_block_status(self._o, count, offset, extent,
                                          completion, flags)

    def aio_get_fd(self):
        '''▶ return file descriptor associated with this connection

    Return the underlying file descriptor associated with
    this connection. You can use this to check if the file
    descriptor is ready for reading or writing and call
    "nbd.aio_notify_read" or "nbd.aio_notify_write". See
    also "nbd.aio_get_direction". Do not do anything else
    with the file descriptor.
'''
        return libnbdmod.aio_get_fd(self._o)

    def aio_get_direction(self):
        '''▶ return the read or write direction

    Return the current direction of this connection, which
    means whether we are next expecting to read data from
    the server, write data to the server, or both. It
    returns

    0   We are not expected to interact with the server file
        descriptor from the current state. It is not worth
        attempting to use poll(2); if the connection is not
        dead, then state machine progress must instead come
        from some other means such as "nbd.aio_connect".

    "AIO_DIRECTION_READ" = 1
        We are expected next to read from the server. If
        using poll(2) you would set "events = POLLIN". If
        "revents" returns "POLLIN" or "POLLHUP" you would
        then call "nbd.aio_notify_read".

        Note that once libnbd reaches "nbd.aio_is_ready",
        this direction is returned even when there are no
        commands in flight (see "nbd.aio_in_flight"). In a
        single-threaded use of libnbd, it is not worth
        polling until after issuing a command, as otherwise
        the server will never wake up the poll. In a
        multi-threaded scenario, you can have one thread
        begin a polling loop prior to any commands, but any
        other thread that issues a command will need a way
        to kick the polling thread out of poll in case
        issuing the command changes the needed polling
        direction. Possible ways to do this include polling
        for activity on a pipe-to-self, or using
        pthread_kill(3) to send a signal that is masked
        except during ppoll(2).

    "AIO_DIRECTION_WRITE" = 2
        We are expected next to write to the server. If
        using poll(2) you would set "events = POLLOUT". If
        "revents" returns "POLLOUT" you would then call
        "nbd.aio_notify_write".

    "AIO_DIRECTION_BOTH" = 3
        We are expected next to either read or write to the
        server. If using poll(2) you would set "events =
        POLLIN|POLLOUT". If only one of "POLLIN" or
        "POLLOUT" is returned, then see above. However, if
        both are returned, it is better to call only
        "nbd.aio_notify_read", as processing the server's
        reply may change the state of the connection and
        invalidate the need to write more commands.
'''
        return libnbdmod.aio_get_direction(self._o)

    def aio_notify_read(self):
        '''▶ notify that the connection is readable

    Send notification to the state machine that the
    connection is readable. Typically this is called after
    your main loop has detected that the file descriptor
    associated with this connection is readable.
'''
        return libnbdmod.aio_notify_read(self._o)

    def aio_notify_write(self):
        '''▶ notify that the connection is writable

    Send notification to the state machine that the
    connection is writable. Typically this is called after
    your main loop has detected that the file descriptor
    associated with this connection is writable.
'''
        return libnbdmod.aio_notify_write(self._o)

    def aio_is_created(self):
        '''▶ check if the connection has just been created

    Return true if this connection has just been created.
    This is the state before the handle has started
    connecting to a server. In this state the handle can
    start to be connected by calling functions such as
    "nbd.aio_connect".
'''
        return libnbdmod.aio_is_created(self._o)

    def aio_is_connecting(self):
        '''▶ check if the connection is connecting or handshaking

    Return true if this connection is connecting to the
    server or in the process of handshaking and negotiating
    options which happens before the handle becomes ready to
    issue commands (see "nbd.aio_is_ready").
'''
        return libnbdmod.aio_is_connecting(self._o)

    def aio_is_negotiating(self):
        '''▶ check if connection is ready to send handshake option

    Return true if this connection is ready to start another
    option negotiation command while handshaking with the
    server. An option command will move back to the
    connecting state (see "nbd.aio_is_connecting"). Note
    that this state cannot be reached unless requested by
    "nbd.set_opt_mode", and even then it only works with
    newstyle servers; an oldstyle server will skip straight
    to "nbd.aio_is_ready".
'''
        return libnbdmod.aio_is_negotiating(self._o)

    def aio_is_ready(self):
        '''▶ check if the connection is in the ready state

    Return true if this connection is connected to the NBD
    server, the handshake has completed, and the connection
    is idle or waiting for a reply. In this state the handle
    is ready to issue commands.
'''
        return libnbdmod.aio_is_ready(self._o)

    def aio_is_processing(self):
        '''▶ check if the connection is processing a command

    Return true if this connection is connected to the NBD
    server, the handshake has completed, and the connection
    is processing commands (either writing out a request or
    reading a reply).

    Note the ready state ("nbd.aio_is_ready") is not
    included. In the ready state commands may be *in flight*
    (the *server* is processing them), but libnbd is not
    processing them.
'''
        return libnbdmod.aio_is_processing(self._o)

    def aio_is_dead(self):
        '''▶ check if the connection is dead

    Return true if the connection has encountered a fatal
    error and is dead. In this state the handle may only be
    closed. There is no way to recover a handle from the
    dead state.
'''
        return libnbdmod.aio_is_dead(self._o)

    def aio_is_closed(self):
        '''▶ check if the connection is closed

    Return true if the connection has closed. There is no
    way to reconnect a closed connection. Instead you must
    close the whole handle.
'''
        return libnbdmod.aio_is_closed(self._o)

    def aio_command_completed(self, cookie):
        '''▶ check if the command completed

    Return true if the command completed. If this function
    returns true then the command was successful and it has
    been retired. Return false if the command is still in
    flight. This can also fail with an error in case the
    command failed (in this case the command is also
    retired). A command is retired either via this command,
    or by using a completion callback which returns 1.

    The "cookie" parameter is the positive unique 64 bit
    cookie for the command, as returned by a call such as
    "nbd.aio_pread".
'''
        return libnbdmod.aio_command_completed(self._o, cookie)

    def aio_peek_command_completed(self):
        '''▶ check if any command has completed

    Return the unique positive 64 bit cookie of the first
    non-retired but completed command, 0 if there are
    in-flight commands but none of them are awaiting
    retirement, or -1 on error including when there are no
    in-flight commands. Any cookie returned by this function
    must still be passed to "nbd.aio_command_completed" to
    actually retire the command and learn whether the
    command was successful.
'''
        return libnbdmod.aio_peek_command_completed(self._o)

    def aio_in_flight(self):
        '''▶ check how many aio commands are still in flight

    Return the number of in-flight aio commands that are
    still awaiting a response from the server before they
    can be retired. If this returns a non-zero value when
    requesting a disconnect from the server (see
    "nbd.aio_disconnect" and "nbd.shutdown"), libnbd does
    not try to wait for those commands to complete
    gracefully; if the server strands commands while
    shutting down, "nbd.aio_command_completed" will report
    those commands as failed with a status of "ENOTCONN".
'''
        return libnbdmod.aio_in_flight(self._o)

    def connection_state(self):
        '''▶ return string describing the state of the connection

    Returns a descriptive string for the state of the
    connection. This can be used for debugging or
    troubleshooting, but you should not rely on the state of
    connections since it may change in future versions.
'''
        return libnbdmod.connection_state(self._o)

    def get_package_name(self):
        '''▶ return the name of the library

    Returns the name of the library, always "libnbd" unless
    the library was modified with another name at compile
    time.
'''
        return libnbdmod.get_package_name(self._o)

    def get_version(self):
        '''▶ return the version of the library

    Return the version of libnbd. This is returned as a
    string in the form "major.minor.release" where each of
    major, minor and release is a small positive integer.
    For example:

         minor
           ↓
        "1.0.3"
         ↑   ↑
     major   release

    major = 0
        The major number was 0 for the early experimental
        versions of libnbd where we still had an unstable
        API.

    major = 1
        The major number is 1 for the versions of libnbd
        with a long-term stable API and ABI. It is not
        anticipated that major will be any number other than
        1.

    minor = 0, 2, ... (even)
        The minor number is even for stable releases.

    minor = 1, 3, ... (odd)
        The minor number is odd for development versions.
        Note that new APIs added in a development version
        remain experimental and subject to change in that
        branch until they appear in a stable release.

    release
        The release number is incremented for each release
        along a particular branch.
'''
        return libnbdmod.get_version(self._o)

    def kill_subprocess(self, signum):
        '''▶ kill server running as a subprocess

    This call may be used to kill the server running as a
    subprocess that was previously created using
    "nbd.connect_command". You do not need to use this call.
    It is only needed if the server does not exit when the
    socket is closed.

    The "signum" parameter is the optional signal number to
    send (see signal(7)). If "signum" is 0 then "SIGTERM" is
    sent.
'''
        return libnbdmod.kill_subprocess(self._o, signum)

    def supports_tls(self):
        '''▶ true if libnbd was compiled with support for TLS

    Returns true if libnbd was compiled with gnutls which is
    required to support TLS encryption, or false if not.
'''
        return libnbdmod.supports_tls(self._o)

    def supports_uri(self):
        '''▶ true if libnbd was compiled with support for NBD URIs

    Returns true if libnbd was compiled with libxml2 which
    is required to support NBD URIs, or false if not.
'''
        return libnbdmod.supports_uri(self._o)

    def get_uri(self):
        '''▶ construct an NBD URI for a connection

    This makes a best effort attempt to construct an NBD URI
    which could be used to connect back to the same server
    (using "nbd.connect_uri").

    In some cases there is not enough information in the
    handle to successfully create a URI (eg. if you
    connected with "nbd.connect_socket"). In such cases the
    call returns "NULL" and further diagnostic information
    is available via "nbd.get_errno" and "nbd.get_error" as
    usual.

    Even if a URI is returned it is not guaranteed to work,
    and it may not be optimal.
'''
        return libnbdmod.get_uri(self._o)


package_name = NBD().get_package_name()
__version__ = NBD().get_version()

if __name__ == "__main__":
    import nbdsh

    nbdsh.shell()
