#!/bin/sh

if test $# -lt 2; then
    cat >> /dev/stderr <<EOF

Usage:

    $0 <repodir> <summarydir>

Track <repodir>s current branch and test each "interesting" commit.
Publish results under <summarydir>.

EOF
    exit 1
fi

set -euvx

webdir=$(dirname $0)

repodir=$1 ; shift
summarydir=$(cd $1 && pwd) ; shift

branch=$(${webdir}/gime-git-branch.sh ${repodir})
remote=$(${webdir}/gime-git-remote.sh ${repodir} ${branch})
start_date=$(date -u -Iseconds)

status() {
    ${webdir}/json-status.sh \
	     --json ${summarydir}/status.json \
	     --job "testing branch: ${branch}" \
	     --start "${start_date}" \
	     --date "$(date -Iseconds -u)" \
	     "$@"
}

status "starting up finding first commit"
start_hash=$(${webdir}/earliest-commit.sh ${repodir} ${summarydir})

status "starting up: building commit list"
${webdir}/json-commit.sh \
	 --json ${summarydir}/commits.json \
	 ${repodir} \
	 $(${webdir}/gime-git-revisions.sh ${repodir} ${start_hash}..${remote}) \
	 ${start_hash}

status "starting up: building summary"
${webdir}/build-summary.sh ${repodir} ${summarydir}

while true ; do

    # Test the existing HEAD.  The intent here is for a restart of
    # this script to resume / retry the most recent test run.
    #
    # Only run the testsuite and update the web site when the current
    # commit looks interesting.  The heuristic is trying to identify
    # coding and testsuite changes; while ignoring infrastructure.

    if ${webdir}/git-interesting.sh ${repodir} HEAD ; then

	status "testing"
	${webdir}/publish.sh ${repodir} ${summarydir}

 	status "updating summary page"
 	${webdir}/build-summary.sh ${repodir} ${summarydir}

    fi

    # Now look for something else to do ...

    while true ; do

	# Time has passed, download any more recent commits, and pull
	# all the updates into ${branch}.  Force ${branch} to be
	# identical to ${remote} by using --ff-only - if it fails the
	# script dies.

	status "fetching ${remote}"
	( cd ${repodir} && git fetch ${remote} || true )
	( cd ${repodir} && git merge --ff-only )

	# Rebuild the commit list so it includes all the just fetched
	# commits.
	#
	# Explicitly include ${start_hash} since that is excluded by
	# the list generated by "git-rev-list".

	status "updating commit list"
	${webdir}/json-commit.sh \
		 --json ${summarydir}/commits.json \
		 ${repodir} \
		 $(${webdir}/gime-git-revisions.sh ${repodir} ${start_hash}..HEAD) \
		 ${start_hash}

	# Starting with HEAD, work backwards looking for anything
	# untested.

	hash=$(${webdir}/gime-work.sh ${summarydir} ${repodir} ${start_hash})
	if test -n "${hash}" ; then
	    status "checking out ${hash}"
	    # Now discard everything back to the commit to be tested,
	    # making that HEAD.  This could have the effect of
	    # switching branches, take care.
	    ( cd ${repodir} && git reset --hard ${hash} )
	    break
	fi

	# Seemlingly nothing todo.

	seconds=$(expr 60 \* 60 \* 3)
	now=$(date +%s)
	future=$(expr ${now} + ${seconds})
	status "idle; will retry $(date -u -d @${future} +%H:%M)"
	sleep ${seconds}

    done

done
