/*
             Common Public License Version 0.5

             THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF
             THIS COMMON PUBLIC LICENSE ("AGREEMENT"). ANY USE,
             REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
             RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.

             1. DEFINITIONS

             "Contribution" means: 
                   a) in the case of the initial Contributor, the
                   initial code and documentation distributed under
                   this Agreement, and 

                   b) in the case of each subsequent Contributor:
                   i) changes to the Program, and
                   ii) additions to the Program;

                   where such changes and/or additions to the Program
                   originate from and are distributed by that
                   particular Contributor. A Contribution 'originates'
                   from a Contributor if it was added to the Program
                   by such Contributor itself or anyone acting on such
                   Contributor's behalf. Contributions do not include
                   additions to the Program which: (i) are separate
                   modules of software distributed in conjunction with
                   the Program under their own license agreement, and
                   (ii) are not derivative works of the Program.


             "Contributor" means any person or entity that distributes
             the Program.

             "Licensed Patents " mean patent claims licensable by a
             Contributor which are necessarily infringed by the use or
             sale of its Contribution alone or when combined with the
             Program. 

             "Program" means the Contributions distributed in
             accordance with this Agreement.

             "Recipient" means anyone who receives the Program under
             this Agreement, including all Contributors.

             2. GRANT OF RIGHTS

                   a) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free copyright
                   license to reproduce, prepare derivative works of,
                   publicly display, publicly perform, distribute and
                   sublicense the Contribution of such Contributor, if
                   any, and such derivative works, in source code and
                   object code form.

                   b) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free patent
                   license under Licensed Patents to make, use, sell,
                   offer to sell, import and otherwise transfer the
                   Contribution of such Contributor, if any, in source
                   code and object code form. This patent license
                   shall apply to the combination of the Contribution
                   and the Program if, at the time the Contribution is
                   added by the Contributor, such addition of the
                   Contribution causes such combination to be covered
                   by the Licensed Patents. The patent license shall
                   not apply to any other combinations which include
                   the Contribution. No hardware per se is licensed
                   hereunder.

                   c) Recipient understands that although each
                   Contributor grants the licenses to its
                   Contributions set forth herein, no assurances are
                   provided by any Contributor that the Program does
                   not infringe the patent or other intellectual
                   property rights of any other entity. Each
                   Contributor disclaims any liability to Recipient
                   for claims brought by any other entity based on
                   infringement of intellectual property rights or
                   otherwise. As a condition to exercising the rights
                   and licenses granted hereunder, each Recipient
                   hereby assumes sole responsibility to secure any
                   other intellectual property rights needed, if any.

                   For example, if a third party patent license is
                   required to allow Recipient to distribute the
                   Program, it is Recipient's responsibility to
                   acquire that license before distributing the
                   Program.

                   d) Each Contributor represents that to its
                   knowledge it has sufficient copyright rights in its
                   Contribution, if any, to grant the copyright
                   license set forth in this Agreement.

             3. REQUIREMENTS

             A Contributor may choose to distribute the Program in
             object code form under its own license agreement, provided
             that:
                   a) it complies with the terms and conditions of
                   this Agreement; and

                   b) its license agreement:
                   i) effectively disclaims on behalf of all
                   Contributors all warranties and conditions, express
                   and implied, including warranties or conditions of
                   title and non-infringement, and implied warranties
                   or conditions of merchantability and fitness for a
                   particular purpose;

                   ii) effectively excludes on behalf of all
                   Contributors all liability for damages, including
                   direct, indirect, special, incidental and
                   consequential damages, such as lost profits;

                   iii) states that any provisions which differ from
                   this Agreement are offered by that Contributor
                   alone and not by any other party; and

                   iv) states that source code for the Program is
                   available from such Contributor, and informs
                   licensees how to obtain it in a reasonable manner
                   on or through a medium customarily used for
                   software exchange.

             When the Program is made available in source code form:
                   a) it must be made available under this Agreement;
                   and
                   b) a copy of this Agreement must be included with
                   each copy of the Program. 

             Contributors may not remove or alter any copyright notices
             contained within the Program.

             Each Contributor must identify itself as the originator of
             its Contribution, if any, in a manner that reasonably
             allows subsequent Recipients to identify the originator of
             the Contribution. 


             4. COMMERCIAL DISTRIBUTION

             Commercial distributors of software may accept certain
             responsibilities with respect to end users, business
             partners and the like. While this license is intended to
             facilitate the commercial use of the Program, the
             Contributor who includes the Program in a commercial
             product offering should do so in a manner which does not
             create potential liability for other Contributors.
             Therefore, if a Contributor includes the Program in a
             commercial product offering, such Contributor ("Commercial
             Contributor") hereby agrees to defend and indemnify every
             other Contributor ("Indemnified Contributor") against any
             losses, damages and costs (collectively "Losses") arising
             from claims, lawsuits and other legal actions brought by a
             third party against the Indemnified Contributor to the
             extent caused by the acts or omissions of such Commercial
             Contributor in connection with its distribution of the
             Program in a commercial product offering. The obligations
             in this section do not apply to any claims or Losses
             relating to any actual or alleged intellectual property
             infringement. In order to qualify, an Indemnified
             Contributor must: a) promptly notify the Commercial
             Contributor in writing of such claim, and b) allow the
             Commercial Contributor to control, and cooperate with the
             Commercial Contributor in, the defense and any related
             settlement negotiations. The Indemnified Contributor may
             participate in any such claim at its own expense.


             For example, a Contributor might include the Program in a
             commercial product offering, Product X. That Contributor
             is then a Commercial Contributor. If that Commercial
             Contributor then makes performance claims, or offers
             warranties related to Product X, those performance claims
             and warranties are such Commercial Contributor's
             responsibility alone. Under this section, the Commercial
             Contributor would have to defend claims against the other
             Contributors related to those performance claims and
             warranties, and if a court requires any other Contributor
             to pay any damages as a result, the Commercial Contributor
             must pay those damages.


             5. NO WARRANTY

             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, THE
             PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT
             WARRANTIES OR CONDITIONS OF ANY KIND, EITHER EXPRESS OR
             IMPLIED INCLUDING, WITHOUT LIMITATION, ANY WARRANTIES OR
             CONDITIONS OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR
             FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is solely
             responsible for determining the appropriateness of using
             and distributing the Program and assumes all risks
             associated with its exercise of rights under this
             Agreement, including but not limited to the risks and
             costs of program errors, compliance with applicable laws,
             damage to or loss of data, programs or equipment, and
             unavailability or interruption of operations. 

             6. DISCLAIMER OF LIABILITY
             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, NEITHER
             RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY
             FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
             OR CONSEQUENTIAL DAMAGES (INCLUDING WITHOUT LIMITATION
             LOST PROFITS), HOWEVER CAUSED AND ON ANY THEORY OF
             LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
             (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
             OF THE USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE
             OF ANY RIGHTS GRANTED HEREUNDER, EVEN IF ADVISED OF THE
             POSSIBILITY OF SUCH DAMAGES.

             7. GENERAL

             If any provision of this Agreement is invalid or
             unenforceable under applicable law, it shall not affect
             the validity or enforceability of the remainder of the
             terms of this Agreement, and without further action by the
             parties hereto, such provision shall be reformed to the
             minimum extent necessary to make such provision valid and
             enforceable.


             If Recipient institutes patent litigation against a
             Contributor with respect to a patent applicable to
             software (including a cross-claim or counterclaim in a
             lawsuit), then any patent licenses granted by that
             Contributor to such Recipient under this Agreement shall
             terminate as of the date such litigation is filed. In
             addition, If Recipient institutes patent litigation
             against any entity (including a cross-claim or
             counterclaim in a lawsuit) alleging that the Program
             itself (excluding combinations of the Program with other
             software or hardware) infringes such Recipient's
             patent(s), then such Recipient's rights granted under
             Section 2(b) shall terminate as of the date such
             litigation is filed.

             All Recipient's rights under this Agreement shall
             terminate if it fails to comply with any of the material
             terms or conditions of this Agreement and does not cure
             such failure in a reasonable period of time after becoming
             aware of such noncompliance. If all Recipient's rights
             under this Agreement terminate, Recipient agrees to cease
             use and distribution of the Program as soon as reasonably
             practicable. However, Recipient's obligations under this
             Agreement and any licenses granted by Recipient relating
             to the Program shall continue and survive. 

             Everyone is permitted to copy and distribute copies of
             this Agreement, but in order to avoid inconsistency the
             Agreement is copyrighted and may only be modified in the
             following manner. The Agreement Steward reserves the right
             to publish new versions (including revisions) of this
             Agreement from time to time. No one other than the
             Agreement Steward has the right to modify this Agreement.

             IBM is the initial Agreement Steward. IBM may assign the
             responsibility to serve as the Agreement Steward to a
             suitable separate entity. Each new version of the
             Agreement will be given a distinguishing version number.
             The Program (including Contributions) may always be
             distributed subject to the version of the Agreement under
             which it was received. In addition, after a new version of
             the Agreement is published, Contributor may elect to
             distribute the Program (including its Contributions) under
             the new version. Except as expressly stated in Sections
             2(a) and 2(b) above, Recipient receives no rights or
             licenses to the intellectual property of any Contributor
             under this Agreement, whether expressly, by implication,
             estoppel or otherwise. All rights in the Program not
             expressly granted under this Agreement are reserved.


             This Agreement is governed by the laws of the State of New
             York and the intellectual property laws of the United
             States of America. No party to this Agreement will bring a
             legal action under this Agreement more than one year after
             the cause of action arose. Each party waives its rights to
             a jury trial in any resulting litigation. 



*/

/* (C) COPYRIGHT International Business Machines Corp. 2001,2002          */


// File:  mech_sha.c
//
// Mechanisms for SHA-1 related routines
//
// The following applies to the software SHA implementation:
//    Written 2 September 1992, Peter C. Gutmann.
//    This implementation placed in the public domain.
//
//    Modified 1 June 1993, Colin Plumb.
//    Modified for the new SHS based on Peter Gutmann's work,
//    18 July 1994, Colin Plumb.
//    Gutmann's work.
//    Renamed to SHA and comments updated a bit 1 November 1995, Colin Plumb.
//    These modifications placed in the public domain.
//
//    Comments to pgut1@cs.aukuni.ac.nz
//

#include <pthread.h>
#include <string.h>            // for memcmp() et al
#include <stdlib.h>
#include <memory.h>

#include "pkcs11types.h"
#include "defs.h"
#include "host_defs.h"
#include "h_extern.h"
#include "tok_spec_struct.h"

#define SHA_HARDWARE_THRESHHOLD 128000

// The SHA f()-functions.  The f1 and f3 functions can be optimized to
// save one boolean operation each - thanks to Rich Schroeppel,
// rcs@cs.arizona.edu for discovering this
//
#define f1(x,y,z) (z ^ (x & (y ^ z)))        // Rounds  0-19
#define f2(x,y,z) (x ^ y ^ z)                // Rounds 20-39
#define f3(x,y,z) ((x & y) | (z & (x | y)))  // Rounds 40-59
#define f4(x,y,z) (x ^ y ^ z)                // Rounds 60-79

// The SHA Mysterious Constants.
// K1 = floor(sqrt(2)  * 2^30)
// K2 = floor(sqrt(3)  * 2^30)
// K3 = floor(sqrt(5)  * 2^30)
// K4 = floor(sqrt(10) * 2^30)
//
#define K1  0x5A827999L // Rounds  0-19
#define K2  0x6ED9EBA1L // Rounds 20-39
#define K3  0x8F1BBCDCL // Rounds 40-59
#define K4  0xCA62C1D6L // Rounds 60-79

// SHA initial values
//
#define h0init 0x67452301
#define h1init 0xEFCDAB89
#define h2init 0x98BADCFE
#define h3init 0x10325476
#define h4init 0xC3D2E1F0

//
// Note that it may be necessary to add parentheses to these macros
// if they are to be called with expressions as arguments.
//

// 32-bit rotate left - kludged with shifts
//

#define ROTL(n,X)  ((X << n) | (X >> (32-n)))

// The initial expanding function
//
// The hash function is defined over an 80-word expanded input array W,
// where the first 16 are copies of the input data, and the remaining 64
// are defined by W[i] = W[i-16] ^ W[i-14] ^ W[i-8] ^ W[i-3].  This
// implementation generates these values on the fly in a circular buffer.
//

#define expand(W,i) \
   (W[i&15] ^= W[(i-14)&15] ^ W[(i-8)&15] ^ W[(i-3)&15], W[i&15] = ROTL(1, W[i&15]))

// The prototype SHA sub-round
//
// The fundamental sub-round is
// a' = e + ROTL(5,a) + f(b, c, d) + k + data;
// b' = a;
// c' = ROTL(30,b);
// d' = c;
// e' = d;
// ... but this is implemented by unrolling the loop 5 times and renaming
// the variables (e,a,b,c,d) = (a',b',c',d',e') each iteration.
//
#define subRound(a, b, c, d, e, f, k, data)  \
   (e += ROTL(5,a) + f(b, c, d) + k + data, b = ROTL(30, b))

/*
 * Perform the SHA transformation.  Note that this code, like MD5, seems to
 * break some optimizing compilers due to the complexity of the expressions
 * and the size of the basic block.  It may be necessary to split it into
 * sections, e.g. based on the four subrounds
 *
 * Note that this corrupts the sha->data area
 */
void shaTransform(SHA1_CONTEXT *ctx)
{
	register unsigned int A, B, C, D, E;

	// Set up first buffer
	//
	A = ctx->hash_value[0];
	B = ctx->hash_value[1];
	C = ctx->hash_value[2];
	D = ctx->hash_value[3];
	E = ctx->hash_value[4];

	// Heavy mangling, in 4 sub-rounds of 20 interations each.
	//
	subRound(A, B, C, D, E, f1, K1, ctx->buf[ 0]);
	subRound(E, A, B, C, D, f1, K1, ctx->buf[ 1]);
	subRound(D, E, A, B, C, f1, K1, ctx->buf[ 2]);
	subRound(C, D, E, A, B, f1, K1, ctx->buf[ 3]);
	subRound(B, C, D, E, A, f1, K1, ctx->buf[ 4]);
	subRound(A, B, C, D, E, f1, K1, ctx->buf[ 5]);
	subRound(E, A, B, C, D, f1, K1, ctx->buf[ 6]);
	subRound(D, E, A, B, C, f1, K1, ctx->buf[ 7]);
	subRound(C, D, E, A, B, f1, K1, ctx->buf[ 8]);
	subRound(B, C, D, E, A, f1, K1, ctx->buf[ 9]);
	subRound(A, B, C, D, E, f1, K1, ctx->buf[10]);
	subRound(E, A, B, C, D, f1, K1, ctx->buf[11]);
	subRound(D, E, A, B, C, f1, K1, ctx->buf[12]);
	subRound(C, D, E, A, B, f1, K1, ctx->buf[13]);
	subRound(B, C, D, E, A, f1, K1, ctx->buf[14]);
	subRound(A, B, C, D, E, f1, K1, ctx->buf[15]);
	subRound(E, A, B, C, D, f1, K1, expand(ctx->buf, 16));
	subRound(D, E, A, B, C, f1, K1, expand(ctx->buf, 17));
	subRound(C, D, E, A, B, f1, K1, expand(ctx->buf, 18));
	subRound(B, C, D, E, A, f1, K1, expand(ctx->buf, 19));

	subRound(A, B, C, D, E, f2, K2, expand(ctx->buf, 20));
	subRound(E, A, B, C, D, f2, K2, expand(ctx->buf, 21));
	subRound(D, E, A, B, C, f2, K2, expand(ctx->buf, 22));
	subRound(C, D, E, A, B, f2, K2, expand(ctx->buf, 23));
	subRound(B, C, D, E, A, f2, K2, expand(ctx->buf, 24));
	subRound(A, B, C, D, E, f2, K2, expand(ctx->buf, 25));
	subRound(E, A, B, C, D, f2, K2, expand(ctx->buf, 26));
	subRound(D, E, A, B, C, f2, K2, expand(ctx->buf, 27));
	subRound(C, D, E, A, B, f2, K2, expand(ctx->buf, 28));
	subRound(B, C, D, E, A, f2, K2, expand(ctx->buf, 29));
	subRound(A, B, C, D, E, f2, K2, expand(ctx->buf, 30));
	subRound(E, A, B, C, D, f2, K2, expand(ctx->buf, 31));
	subRound(D, E, A, B, C, f2, K2, expand(ctx->buf, 32));
	subRound(C, D, E, A, B, f2, K2, expand(ctx->buf, 33));
	subRound(B, C, D, E, A, f2, K2, expand(ctx->buf, 34));
	subRound(A, B, C, D, E, f2, K2, expand(ctx->buf, 35));
	subRound(E, A, B, C, D, f2, K2, expand(ctx->buf, 36));
	subRound(D, E, A, B, C, f2, K2, expand(ctx->buf, 37));
	subRound(C, D, E, A, B, f2, K2, expand(ctx->buf, 38));
	subRound(B, C, D, E, A, f2, K2, expand(ctx->buf, 39));

	subRound(A, B, C, D, E, f3, K3, expand(ctx->buf, 40));
	subRound(E, A, B, C, D, f3, K3, expand(ctx->buf, 41));
	subRound(D, E, A, B, C, f3, K3, expand(ctx->buf, 42));
	subRound(C, D, E, A, B, f3, K3, expand(ctx->buf, 43));
	subRound(B, C, D, E, A, f3, K3, expand(ctx->buf, 44));
	subRound(A, B, C, D, E, f3, K3, expand(ctx->buf, 45));
	subRound(E, A, B, C, D, f3, K3, expand(ctx->buf, 46));
	subRound(D, E, A, B, C, f3, K3, expand(ctx->buf, 47));
	subRound(C, D, E, A, B, f3, K3, expand(ctx->buf, 48));
	subRound(B, C, D, E, A, f3, K3, expand(ctx->buf, 49));
	subRound(A, B, C, D, E, f3, K3, expand(ctx->buf, 50));
	subRound(E, A, B, C, D, f3, K3, expand(ctx->buf, 51));
	subRound(D, E, A, B, C, f3, K3, expand(ctx->buf, 52));
	subRound(C, D, E, A, B, f3, K3, expand(ctx->buf, 53));
	subRound(B, C, D, E, A, f3, K3, expand(ctx->buf, 54));
	subRound(A, B, C, D, E, f3, K3, expand(ctx->buf, 55));
	subRound(E, A, B, C, D, f3, K3, expand(ctx->buf, 56));
	subRound(D, E, A, B, C, f3, K3, expand(ctx->buf, 57));
	subRound(C, D, E, A, B, f3, K3, expand(ctx->buf, 58));
	subRound(B, C, D, E, A, f3, K3, expand(ctx->buf, 59));

	subRound(A, B, C, D, E, f4, K4, expand(ctx->buf, 60));
	subRound(E, A, B, C, D, f4, K4, expand(ctx->buf, 61));
	subRound(D, E, A, B, C, f4, K4, expand(ctx->buf, 62) );
	subRound(C, D, E, A, B, f4, K4, expand(ctx->buf, 63) );
	subRound(B, C, D, E, A, f4, K4, expand(ctx->buf, 64) );
	subRound(A, B, C, D, E, f4, K4, expand(ctx->buf, 65) );
	subRound(E, A, B, C, D, f4, K4, expand(ctx->buf, 66) );
	subRound(D, E, A, B, C, f4, K4, expand(ctx->buf, 67) );
	subRound(C, D, E, A, B, f4, K4, expand(ctx->buf, 68) );
	subRound(B, C, D, E, A, f4, K4, expand(ctx->buf, 69) );
	subRound(A, B, C, D, E, f4, K4, expand(ctx->buf, 70) );
	subRound(E, A, B, C, D, f4, K4, expand(ctx->buf, 71) );
	subRound(D, E, A, B, C, f4, K4, expand(ctx->buf, 72) );
	subRound(C, D, E, A, B, f4, K4, expand(ctx->buf, 73) );
	subRound(B, C, D, E, A, f4, K4, expand(ctx->buf, 74) );
	subRound(A, B, C, D, E, f4, K4, expand(ctx->buf, 75) );
	subRound(E, A, B, C, D, f4, K4, expand(ctx->buf, 76) );
	subRound(D, E, A, B, C, f4, K4, expand(ctx->buf, 77) );
	subRound(C, D, E, A, B, f4, K4, expand(ctx->buf, 78) );
	subRound(B, C, D, E, A, f4, K4, expand(ctx->buf, 79) );

	// Build message digest
	//
	ctx->hash_value[0] += A;
	ctx->hash_value[1] += B;
	ctx->hash_value[2] += C;
	ctx->hash_value[3] += D;
	ctx->hash_value[4] += E;
}

/*
* SHA is defined in big-endian form, so this converts the buffer from
* bytes to words, independent of the machine's native endianness.
*
* Assuming a consistent byte ordering for the machine, this also
* has the magic property of being self-inverse.  It is used as
* such.
*/
static void byteReverse(unsigned int *buffer, unsigned int byteCount)
{
#ifndef __BYTE_ORDER
#error  "Endianess MUST be defined"
#endif
#if  __BYTE_ORDER == __LITTLE_ENDIAN
	CK_ULONG value, val;

	byteCount /= sizeof(CK_ULONG_32);

	while (byteCount--) {
		val = *buffer;
		value = ((0x000000FF & val) << 24) |
			((0x0000FF00 & val) << 8 ) |
			((0x00FF0000 & val) >> 8 ) |
			((0xFF000000 & val) >> 24);

		*buffer++ = value;
	}
#endif

	/*
	 * JRM - this code gives funky results on Linux/Intel.
	 * I assume this is a GCC issue since regression tests passed on NT
	 *
	 * byteCount /= sizeof(CK_ULONG);
	 * while ( byteCount-- ) {
	 * 	value = (CK_ULONG)((unsigned)((CK_BYTE *)buffer)[0] << 8 |
	 *	    ((CK_BYTE *)buffer)[1]) << 16 |
	 *	    ((unsigned)((CK_BYTE *)buffer)[2] << 8 |
	 *	    ((CK_BYTE *)buffer)[3]);
	 * 	*buffer++ = value;
	 * }
	 */
}


void shaUpdate(SHA1_CONTEXT *ctx, CK_BYTE const *buffer, CK_ULONG count)
{
	CK_ULONG t;

	// Update bitcount
	//
	t = ctx->bits_lo;
	if ((ctx->bits_lo = t + count) < t)
		ctx->bits_hi++;   // Carry from low to high

	t &= 0x3f;  // Bytes already in ctx->buf

	// Handle any leading odd-sized chunks
	//
	if (t) {
		CK_BYTE *p = (CK_BYTE *)ctx->buf + t;

		t = 64-t;
		if (count < t) {
			memcpy(p, buffer, count);
			return;
		}
		memcpy(p, buffer, t);
		byteReverse(ctx->buf, SHA1_BLOCK_SIZE);
		shaTransform(ctx);
		buffer += t;
		count -= t;
	}

	// Process data in SHA1_BLOCK_SIZE chunks
	//
	while (count >= SHA1_BLOCK_SIZE) {
		memcpy(ctx->buf, buffer, SHA1_BLOCK_SIZE);
		byteReverse(ctx->buf, SHA1_BLOCK_SIZE);
		shaTransform(ctx);
		buffer += SHA1_BLOCK_SIZE;
		count -= SHA1_BLOCK_SIZE;
	}

	// Handle any remaining bytes of data.
	//
	memcpy(ctx->buf, buffer, count);
}


/*
 * Final wrapup - pad to 64-byte boundary with the bit pattern
 * 1 0* (64-bit count of bits processed, MSB-first)
 */
void shaFinal(SHA1_CONTEXT *ctx, CK_BYTE *hash)
{
	int count;
	CK_BYTE *p;

	// Compute number of bytes mod 64
	//
	count = (int)ctx->bits_lo & 0x3F;

	// Set the first char of padding to 0x80.
	// This is safe since there is always at least one byte free
	//
	p = (CK_BYTE *)ctx->buf + count;
	*p++ = 0x80;

	// Bytes of padding needed to make 64 bytes
	//
	count = SHA1_BLOCK_SIZE - 1 - count;

	// Pad out to 56 mod 64
	//
	if (count < 8) {
		// Two lots of padding:  Pad the first block to 64 bytes
		//
		memset(p, 0, count);
		byteReverse(ctx->buf, SHA1_BLOCK_SIZE);
		shaTransform(ctx);

		// Now fill the next block with 56 bytes
		//
		memset(ctx->buf, 0, SHA1_BLOCK_SIZE-8);
	} else {
		// Pad block to 56 bytes
		//
		memset(p, 0, count-8);
	}
	byteReverse(ctx->buf, SHA1_BLOCK_SIZE-8);

	// Append length in *bits* and transform
	//
	ctx->buf[14] = ctx->bits_hi << 3 | ctx->bits_lo >> 29;
	ctx->buf[15] = ctx->bits_lo << 3;

	shaTransform(ctx);

	// Store output hash in buffer
	//
	byteReverse(ctx->hash_value, SHA1_HASH_SIZE);
	memcpy(hash, ctx->hash_value, SHA1_HASH_SIZE);
}

//
// Software SHA-1 implementation
//

void sw_sha1_init(DIGEST_CONTEXT *ctx)
{
	// Set the h-vars to their initial values
	SHA1_CONTEXT *sha1_ctx;
	/* Allocate the context */
	ctx->context_len = sizeof(SHA1_CONTEXT);
	ctx->context = (CK_BYTE *)malloc(sizeof(SHA1_CONTEXT));
	if (ctx->context == NULL) {
		OCK_LOG_ERR(ERR_HOST_MEMORY);
		// TODO: propagate error up?
		return;
	}

	sha1_ctx = (SHA1_CONTEXT *)ctx->context;
	sha1_ctx->hash_value[0]  = h0init;
	sha1_ctx->hash_value[1]  = h1init;
	sha1_ctx->hash_value[2]  = h2init;
	sha1_ctx->hash_value[3]  = h3init;
	sha1_ctx->hash_value[4]  = h4init;

	// Initialise bit count
 	sha1_ctx->bits_lo = sha1_ctx->bits_hi = 0;
}

CK_RV sw_sha1_hash(DIGEST_CONTEXT *ctx, CK_BYTE *in_data, CK_ULONG in_data_len,
		   CK_BYTE *out_data, CK_ULONG *out_data_len)
{
	CK_RV rv;
   
	if (!ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (*out_data_len < SHA1_HASH_SIZE) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (ctx->context == NULL)
		return CKR_HOST_MEMORY;

	shaUpdate((SHA1_CONTEXT *)ctx->context, in_data, in_data_len);
	shaFinal((SHA1_CONTEXT *)ctx->context, out_data);
	*out_data_len = SHA1_HASH_SIZE;
	return CKR_OK;
}

CK_RV sha1_hash(SESSION *sess, CK_BBOOL length_only, DIGEST_CONTEXT *ctx,
		CK_BYTE *in_data, CK_ULONG in_data_len, CK_BYTE *out_data,
		CK_ULONG *out_data_len)
{
	CK_RV rv;
   
	if (!ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (length_only == TRUE) {
		*out_data_len = SHA1_HASH_SIZE;
		return CKR_OK;
	}

	if (*out_data_len < SHA1_HASH_SIZE) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (ctx->context == NULL)
		return CKR_HOST_MEMORY;

	if (token_specific.t_sha == NULL ) {
		return sw_sha1_hash(ctx, in_data, in_data_len, out_data,
			     out_data_len);
	} else 
		return token_specific.t_sha(ctx, in_data, in_data_len,
						out_data, out_data_len);
}

CK_RV sha2_hash(SESSION *sess, CK_BBOOL length_only, DIGEST_CONTEXT *ctx,
		CK_BYTE *in_data, CK_ULONG in_data_len, CK_BYTE *out_data,
		CK_ULONG *out_data_len)
{
	CK_RV rv;

	if (!ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (length_only == TRUE) {
		*out_data_len = SHA2_HASH_SIZE;
		return CKR_OK;
	}

	if (ctx->context == NULL)
	return CKR_HOST_MEMORY;

	if (token_specific.t_sha2 == NULL) 
        	return CKR_MECHANISM_INVALID;
	else
    		return token_specific.t_sha2(ctx, in_data, in_data_len,
						out_data, out_data_len);
}

CK_RV
sha3_hash(SESSION *sess, CK_BBOOL length_only, DIGEST_CONTEXT *ctx,
	  CK_BYTE *in_data, CK_ULONG in_data_len, CK_BYTE *out_data,
	  CK_ULONG *out_data_len)
{
	CK_RV rv;

	if (!ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (length_only == TRUE) {
		*out_data_len = SHA3_HASH_SIZE;
		return CKR_OK;
	}

	if (ctx->context == NULL)
		return CKR_HOST_MEMORY;

	if (token_specific.t_sha3 == NULL)
		return CKR_MECHANISM_INVALID;
	else
    		return token_specific.t_sha3(ctx, in_data, in_data_len,
					     out_data, out_data_len);

}

CK_RV sha5_hash(SESSION *sess, CK_BBOOL length_only, DIGEST_CONTEXT *ctx,
		CK_BYTE *in_data, CK_ULONG in_data_len, CK_BYTE *out_data,
		CK_ULONG *out_data_len)
{
	CK_RV rv;

	if (!ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (length_only == TRUE) {
		*out_data_len = SHA5_HASH_SIZE;
		return CKR_OK;
	}

	if (ctx->context == NULL)
		return CKR_HOST_MEMORY;

	if (token_specific.t_sha5 == NULL)
		return CKR_MECHANISM_INVALID;

	return token_specific.t_sha5(ctx, in_data, in_data_len, out_data, 
					out_data_len);
}

//
//
CK_RV sha1_hash_update(SESSION *sess, DIGEST_CONTEXT *ctx, CK_BYTE *in_data,
		       CK_ULONG in_data_len)
{
	if (!in_data) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (token_specific.t_sha_update == NULL) {
        	shaUpdate((SHA1_CONTEXT *)ctx->context, in_data, in_data_len);
		return CKR_OK;
	} else
		return token_specific.t_sha_update(ctx, in_data, in_data_len);
}

CK_RV sha2_hash_update(SESSION *sess, DIGEST_CONTEXT *ctx, CK_BYTE *in_data,
		       CK_ULONG in_data_len)
{
	if (!in_data) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (token_specific.t_sha2_update == NULL)
		return CKR_MECHANISM_INVALID;
	else
		return token_specific.t_sha2_update(ctx, in_data, in_data_len);
}

CK_RV sha3_hash_update(SESSION *sess, DIGEST_CONTEXT *ctx, CK_BYTE *in_data,
		       CK_ULONG in_data_len)
{
	if (!in_data) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (token_specific.t_sha3_update == NULL)
		return CKR_MECHANISM_INVALID;
    	else
		return token_specific.t_sha3_update(ctx, in_data, in_data_len);
}

CK_RV sha5_hash_update(SESSION *sess, DIGEST_CONTEXT *ctx, CK_BYTE *in_data,
		       CK_ULONG in_data_len)
{
	if (!in_data) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}
	
	if(token_specific.t_sha5_update == NULL)
		return CKR_MECHANISM_INVALID;
	else
		return token_specific.t_sha5_update(ctx, in_data, in_data_len);
}

CK_RV sha1_hash_final(SESSION *sess, CK_BYTE length_only, DIGEST_CONTEXT *ctx,
		      CK_BYTE *out_data, CK_ULONG *out_data_len)
{
	if (!ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (length_only == TRUE) {
		*out_data_len = SHA1_HASH_SIZE;
		return CKR_OK;
	}

        if (*out_data_len < SHA1_HASH_SIZE){
            OCK_LOG_ERR(ERR_FUNCTION_FAILED);
            return CKR_FUNCTION_FAILED;
        }

	if (token_specific.t_sha_final == NULL) {
		shaFinal((SHA1_CONTEXT *)ctx->context, out_data);
		*out_data_len = SHA1_HASH_SIZE;
		return CKR_OK;
	} else
		return token_specific.t_sha_final(ctx, out_data, out_data_len);
}

CK_RV sha2_hash_final(SESSION *sess, CK_BYTE length_only, DIGEST_CONTEXT *ctx,
		      CK_BYTE *out_data, CK_ULONG *out_data_len)
{
	if (!ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}
	if (length_only == TRUE) {
		*out_data_len = SHA2_HASH_SIZE;
		return CKR_OK;
	}

	if (token_specific.t_sha2_final == NULL)
		return CKR_MECHANISM_INVALID;
	else
		return token_specific.t_sha2_final(ctx, out_data, out_data_len);
}

CK_RV sha3_hash_final(SESSION *sess, CK_BYTE length_only, DIGEST_CONTEXT *ctx,
		      CK_BYTE *out_data, CK_ULONG *out_data_len)
{
	if (!ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (length_only == TRUE) {
		*out_data_len = SHA3_HASH_SIZE;
		return CKR_OK;
	}

	if (token_specific.t_sha3_final == NULL)
		return CKR_MECHANISM_INVALID;
	else
		return token_specific.t_sha3_final(ctx, out_data, out_data_len);
}

CK_RV sha5_hash_final(SESSION *sess, CK_BYTE length_only, DIGEST_CONTEXT *ctx,
		      CK_BYTE *out_data, CK_ULONG *out_data_len)
{
	if (!ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (length_only == TRUE) {
		*out_data_len = SHA5_HASH_SIZE;
		return CKR_OK;
	}

	if (token_specific.t_sha5_final == NULL)
		return CKR_MECHANISM_INVALID;
	else
		return token_specific.t_sha5_final(ctx, out_data, out_data_len);
}

// this routine gets called for two mechanisms actually:
//    CKM_SHA_1_HMAC
//    CKM_SHA_1_HMAC_GENERAL
//
CK_RV sha1_hmac_sign(SESSION *sess, CK_BBOOL length_only,
		     SIGN_VERIFY_CONTEXT *ctx, CK_BYTE *in_data,
		     CK_ULONG in_data_len, CK_BYTE *out_data,
		     CK_ULONG *out_data_len)
{
	OBJECT *key_obj = NULL;
	CK_ATTRIBUTE *attr = NULL;
	CK_BYTE hash[SHA1_HASH_SIZE];
	DIGEST_CONTEXT digest_ctx;
	CK_MECHANISM digest_mech;
	CK_BYTE k_ipad[SHA1_BLOCK_SIZE];
	CK_BYTE k_opad[SHA1_BLOCK_SIZE];
	CK_ULONG key_bytes, hash_len, hmac_len;
	CK_ULONG i;
	CK_RV rc;

	if (!sess || !ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (ctx->mech.mechanism == CKM_SHA_1_HMAC_GENERAL) {
		hmac_len = *(CK_ULONG *)ctx->mech.pParameter;

		if (hmac_len == 0) {
			*out_data_len = 0;
			return CKR_OK;
		}
	} else
		hmac_len = SHA1_HASH_SIZE;

	if (length_only == TRUE) {
		*out_data_len = hmac_len;
		return CKR_OK;
	}

	memset(&digest_ctx, 0x0, sizeof(DIGEST_CONTEXT));

	rc = object_mgr_find_in_map1(ctx->key, &key_obj);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_OBJMGR_FIND_MAP);
		return rc;
	}

	rc = template_attribute_find(key_obj->template, CKA_VALUE, &attr);
	if (rc == FALSE) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	} else
		key_bytes = attr->ulValueLen;


	// build (K XOR ipad), (K XOR opad)
	//
	if (key_bytes > SHA1_BLOCK_SIZE) {
		digest_mech.mechanism = CKM_SHA_1;
		digest_mech.ulParameterLen = 0;
		digest_mech.pParameter = NULL;

		rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
		if (rc != CKR_OK) {
			OCK_LOG_ERR(ERR_DIGEST_INIT);
			return rc;
		}

		hash_len = sizeof(hash);
		rc = digest_mgr_digest(sess, FALSE, &digest_ctx, attr->pValue,
				       attr->ulValueLen, hash, &hash_len);
		if (rc != CKR_OK) {
			OCK_LOG_ERR(ERR_DIGEST);
			return rc;
		}

		memset(&digest_ctx, 0x0, sizeof(DIGEST_CONTEXT));

		for (i=0; i < hash_len; i++) {
			k_ipad[i] = hash[i] ^ 0x36;
			k_opad[i] = hash[i] ^ 0x5C;
		}

		memset(&k_ipad[i], 0x36, SHA1_BLOCK_SIZE - i);
		memset(&k_opad[i], 0x5C, SHA1_BLOCK_SIZE - i);
	} else {
		CK_BYTE *key = attr->pValue;

		for (i=0; i < key_bytes; i++) {
			k_ipad[i] = key[i] ^ 0x36;
			k_opad[i] = key[i] ^ 0x5C;
		}

		memset(&k_ipad[i], 0x36, SHA1_BLOCK_SIZE - key_bytes);
		memset(&k_opad[i], 0x5C, SHA1_BLOCK_SIZE - key_bytes);
	}

	digest_mech.mechanism = CKM_SHA_1;
	digest_mech.ulParameterLen = 0;
	digest_mech.pParameter = NULL;

	// inner hash
	//
	rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_INIT);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, k_ipad,
				      SHA1_BLOCK_SIZE);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, in_data, in_data_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	hash_len = sizeof(hash);
	rc = digest_mgr_digest_final(sess, FALSE, &digest_ctx, hash, &hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_FINAL);
		return rc;
	}

	memset(&digest_ctx, 0x0, sizeof(DIGEST_CONTEXT));

	// outer hash
	//
	rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_INIT);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, k_opad,
					SHA1_BLOCK_SIZE);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, hash, hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	hash_len = sizeof(hash);
	rc = digest_mgr_digest_final(sess, FALSE, &digest_ctx, hash, &hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_FINAL);
		return rc;
	}

	memcpy(out_data, hash, hmac_len);
	*out_data_len = hmac_len;

	return CKR_OK;
}

/** This routine gets called for two mechanisms actually:
 *    CKM_SHA256_HMAC
 *    CKM_SHA256_HMAC_GENERAL
 */
CK_RV sha2_hmac_sign(SESSION *sess, CK_BBOOL length_only,
		     SIGN_VERIFY_CONTEXT *ctx, CK_BYTE *in_data,
		     CK_ULONG in_data_len, CK_BYTE *out_data,
		     CK_ULONG *out_data_len)
{
	OBJECT *key_obj = NULL;
	CK_ATTRIBUTE *attr = NULL;
	CK_BYTE hash[SHA2_HASH_SIZE];
	DIGEST_CONTEXT digest_ctx;
	CK_MECHANISM digest_mech;
	CK_BYTE k_ipad[SHA2_BLOCK_SIZE];
	CK_BYTE k_opad[SHA2_BLOCK_SIZE];
	CK_ULONG key_bytes, hash_len, hmac_len;
	CK_ULONG i;
	CK_RV rc;

	if (!sess || !ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (ctx->mech.mechanism == CKM_SHA256_HMAC_GENERAL) {
		hmac_len = *(CK_ULONG *)ctx->mech.pParameter;

		if (hmac_len == 0) {
			*out_data_len = 0;
			return CKR_OK;
		}
	} else
		hmac_len = SHA2_HASH_SIZE;

	if (length_only == TRUE) {
		*out_data_len = hmac_len;
		return CKR_OK;
	}

	memset(&digest_ctx, 0x0, sizeof(DIGEST_CONTEXT));

	rc = object_mgr_find_in_map1(ctx->key, &key_obj);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_OBJMGR_FIND_MAP);
		return rc;
	}
	rc = template_attribute_find(key_obj->template, CKA_VALUE, &attr);
	if (rc == FALSE) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	} else
		key_bytes = attr->ulValueLen;

	// build (K XOR ipad), (K XOR opad)
	//
	if (key_bytes > SHA2_BLOCK_SIZE) {
		digest_mech.mechanism = CKM_SHA256;
		digest_mech.ulParameterLen = 0;
		digest_mech.pParameter = NULL;

		rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
		if (rc != CKR_OK) {
			OCK_LOG_ERR(ERR_DIGEST_INIT);
			return rc;
		}

		hash_len = sizeof(hash);
		rc = digest_mgr_digest(sess, FALSE, &digest_ctx, attr->pValue,
					attr->ulValueLen, hash, &hash_len);
		if (rc != CKR_OK) {
			OCK_LOG_ERR(ERR_DIGEST);
			return rc;
		}

		memset(&digest_ctx, 0x0, sizeof(DIGEST_CONTEXT));

		for (i=0; i < hash_len; i++) {
			k_ipad[i] = hash[i] ^ 0x36;
			k_opad[i] = hash[i] ^ 0x5C;
		}

		memset(&k_ipad[i], 0x36, SHA2_BLOCK_SIZE - i);
		memset(&k_opad[i], 0x5C, SHA2_BLOCK_SIZE - i);
	} else {
		CK_BYTE *key = attr->pValue;

		for (i=0; i < key_bytes; i++) {
			k_ipad[i] = key[i] ^ 0x36;
			k_opad[i] = key[i] ^ 0x5C;
		}

		memset(&k_ipad[i], 0x36, SHA2_BLOCK_SIZE - key_bytes);
		memset(&k_opad[i], 0x5C, SHA2_BLOCK_SIZE - key_bytes);
	}

	digest_mech.mechanism = CKM_SHA256;
	digest_mech.ulParameterLen = 0;
	digest_mech.pParameter = NULL;

	// inner hash
	//
	rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_INIT);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, k_ipad, SHA2_BLOCK_SIZE);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, in_data, in_data_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	hash_len = sizeof(hash);
	rc = digest_mgr_digest_final(sess, FALSE, &digest_ctx, hash, &hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_FINAL);
		return rc;
	}

	memset(&digest_ctx, 0x0, sizeof(DIGEST_CONTEXT));

	// outer hash
	//
	rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_INIT);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, k_opad, SHA2_BLOCK_SIZE);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, hash, hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	hash_len = sizeof(hash);
	rc = digest_mgr_digest_final(sess, FALSE, &digest_ctx, hash, &hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_FINAL);
		return rc;
	}

	memcpy(out_data, hash, hmac_len);
	*out_data_len = hmac_len;

	return CKR_OK;
}

/** This routine gets called for two mechanisms actually:
 *    CKM_SHA384_HMAC
 *    CKM_SHA384_HMAC_GENERAL
 */
CK_RV sha3_hmac_sign(SESSION *sess, CK_BBOOL length_only,
		     SIGN_VERIFY_CONTEXT *ctx, CK_BYTE *in_data,
		     CK_ULONG in_data_len, CK_BYTE *out_data,
		     CK_ULONG *out_data_len)
{
	OBJECT *key_obj = NULL;
	CK_ATTRIBUTE *attr = NULL;
	CK_BYTE hash[SHA3_HASH_SIZE];
	DIGEST_CONTEXT digest_ctx;
	CK_MECHANISM digest_mech;
	CK_BYTE k_ipad[SHA3_BLOCK_SIZE];
	CK_BYTE k_opad[SHA3_BLOCK_SIZE];
	CK_ULONG key_bytes, hash_len, hmac_len;
	CK_ULONG i;
	CK_RV rc;

	if (!sess || !ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (ctx->mech.mechanism == CKM_SHA384_HMAC_GENERAL) {
		hmac_len = *(CK_ULONG *)ctx->mech.pParameter;

		if (hmac_len == 0) {
			*out_data_len = 0;
			return CKR_OK;
		}
   	} else
		hmac_len = SHA3_HASH_SIZE;

	if (length_only == TRUE) {
		*out_data_len = hmac_len;
		return CKR_OK;
	}

	memset( &digest_ctx, 0x0, sizeof(DIGEST_CONTEXT) );

	rc = object_mgr_find_in_map1( ctx->key, &key_obj );
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_OBJMGR_FIND_MAP);
		return rc;
	}
	rc = template_attribute_find(key_obj->template, CKA_VALUE, &attr);
	if (rc == FALSE) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	} else
		key_bytes = attr->ulValueLen;

	// build (K XOR ipad), (K XOR opad)
	//
	if (key_bytes > SHA3_BLOCK_SIZE) {
		digest_mech.mechanism = CKM_SHA384;
		digest_mech.ulParameterLen = 0;
		digest_mech.pParameter = NULL;

		rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
		if (rc != CKR_OK) {
			OCK_LOG_ERR(ERR_DIGEST_INIT);
			return rc;
		}

		hash_len = sizeof(hash);
		rc = digest_mgr_digest(sess, FALSE, &digest_ctx, attr->pValue,
					attr->ulValueLen, hash, &hash_len);
		if (rc != CKR_OK) {
			OCK_LOG_ERR(ERR_DIGEST);
			return rc;
		}

		memset(&digest_ctx, 0x0, sizeof(DIGEST_CONTEXT));

		for (i=0; i < hash_len; i++) {
			k_ipad[i] = hash[i] ^ 0x36;
			k_opad[i] = hash[i] ^ 0x5C;
		}

		memset(&k_ipad[i], 0x36, SHA3_BLOCK_SIZE - i);
		memset(&k_opad[i], 0x5C, SHA3_BLOCK_SIZE - i);
	} else {
		CK_BYTE *key = attr->pValue;

		for (i=0; i < key_bytes; i++) {
			k_ipad[i] = key[i] ^ 0x36;
			k_opad[i] = key[i] ^ 0x5C;
		}

		memset(&k_ipad[i], 0x36, SHA3_BLOCK_SIZE - key_bytes);
		memset(&k_opad[i], 0x5C, SHA3_BLOCK_SIZE - key_bytes);
	}

	digest_mech.mechanism = CKM_SHA384;
	digest_mech.ulParameterLen = 0;
	digest_mech.pParameter = NULL;

	// inner hash
	//
	rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_INIT);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, k_ipad, SHA3_BLOCK_SIZE);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, in_data, in_data_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	hash_len = sizeof(hash);
	rc = digest_mgr_digest_final(sess, FALSE, &digest_ctx, hash, &hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_FINAL);
		return rc;
	}

	memset(&digest_ctx, 0x0, sizeof(DIGEST_CONTEXT));

	// outer hash
	//
	rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_INIT);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, k_opad, SHA3_BLOCK_SIZE);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, hash, hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	hash_len = sizeof(hash);
	rc = digest_mgr_digest_final(sess, FALSE, &digest_ctx, hash, &hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_FINAL);
		return rc;
	}

	memcpy(out_data, hash, hmac_len);
	*out_data_len = hmac_len;

	return CKR_OK;
}

/** This routine gets called for two mechanisms actually:
 *    CKM_SHA512_HMAC
 *    CKM_SHA512_HMAC_GENERAL
 */
CK_RV sha5_hmac_sign(SESSION *sess, CK_BBOOL length_only,
		     SIGN_VERIFY_CONTEXT *ctx, CK_BYTE *in_data,
		     CK_ULONG in_data_len, CK_BYTE *out_data,
		     CK_ULONG *out_data_len)
{
	OBJECT *key_obj = NULL;
	CK_ATTRIBUTE *attr = NULL;
	CK_BYTE hash[SHA5_HASH_SIZE];
	DIGEST_CONTEXT digest_ctx;
	CK_MECHANISM digest_mech;
	CK_BYTE k_ipad[SHA5_BLOCK_SIZE];
	CK_BYTE k_opad[SHA5_BLOCK_SIZE];
	CK_ULONG key_bytes, hash_len, hmac_len;
	CK_ULONG i;
	CK_RV rc;

	if (!sess || !ctx || !out_data_len) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (ctx->mech.mechanism == CKM_SHA512_HMAC_GENERAL) {
		hmac_len = *(CK_ULONG *)ctx->mech.pParameter;

		if (hmac_len == 0) {
			*out_data_len = 0;
			return CKR_OK;
		}
	} else
		hmac_len = SHA5_HASH_SIZE;

	if (length_only == TRUE) {
		*out_data_len = hmac_len;
		return CKR_OK;
	}

	memset(&digest_ctx, 0x0, sizeof(DIGEST_CONTEXT));

	rc = object_mgr_find_in_map1(ctx->key, &key_obj);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_OBJMGR_FIND_MAP);
		return rc;
	}
	rc = template_attribute_find(key_obj->template, CKA_VALUE, &attr);
	if (rc == FALSE) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	} else
		key_bytes = attr->ulValueLen;

	// build (K XOR ipad), (K XOR opad)
	//
	if (key_bytes > SHA5_BLOCK_SIZE) {
		digest_mech.mechanism = CKM_SHA512;
		digest_mech.ulParameterLen = 0;
		digest_mech.pParameter = NULL;

		rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
		if (rc != CKR_OK) {
			OCK_LOG_ERR(ERR_DIGEST_INIT);
			return rc;
		}

		hash_len = sizeof(hash);
		rc = digest_mgr_digest(sess, FALSE, &digest_ctx, attr->pValue,
					attr->ulValueLen, hash, &hash_len);
		if (rc != CKR_OK) {
			OCK_LOG_ERR(ERR_DIGEST);
			return rc;
		}

		memset(&digest_ctx, 0x0, sizeof(DIGEST_CONTEXT));

		for (i=0; i < hash_len; i++) {
			k_ipad[i] = hash[i] ^ 0x36;
			k_opad[i] = hash[i] ^ 0x5C;
		}

		memset(&k_ipad[i], 0x36, SHA5_BLOCK_SIZE - i);
		memset(&k_opad[i], 0x5C, SHA5_BLOCK_SIZE - i);
	} else {
		CK_BYTE *key = attr->pValue;

		for (i=0; i < key_bytes; i++) {
			k_ipad[i] = key[i] ^ 0x36;
			k_opad[i] = key[i] ^ 0x5C;
		}

		memset(&k_ipad[i], 0x36, SHA5_BLOCK_SIZE - key_bytes);
		memset(&k_opad[i], 0x5C, SHA5_BLOCK_SIZE - key_bytes);
	}

	digest_mech.mechanism = CKM_SHA512;
	digest_mech.ulParameterLen = 0;
	digest_mech.pParameter = NULL;

	// inner hash
	//
	rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_INIT);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, k_ipad, SHA5_BLOCK_SIZE);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, in_data, in_data_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	hash_len = sizeof(hash);
	rc = digest_mgr_digest_final(sess, FALSE, &digest_ctx, hash, &hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_FINAL);
		return rc;
	}

	memset( &digest_ctx, 0x0, sizeof(DIGEST_CONTEXT) );

	// outer hash
	//
	rc = digest_mgr_init(sess, &digest_ctx, &digest_mech);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_INIT);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, k_opad, SHA5_BLOCK_SIZE);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	rc = digest_mgr_digest_update(sess, &digest_ctx, hash, hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_UPDATE);
		return rc;
	}

	hash_len = sizeof(hash);
	rc = digest_mgr_digest_final(sess, FALSE, &digest_ctx, hash, &hash_len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_DIGEST_FINAL);
		return rc;
	}

	memcpy(out_data, hash, hmac_len);
	*out_data_len = hmac_len;

	return CKR_OK;
}

CK_RV sha1_hmac_verify(SESSION *sess, SIGN_VERIFY_CONTEXT *ctx,
		       CK_BYTE *in_data, CK_ULONG in_data_len,
		       CK_BYTE *signature, CK_ULONG sig_len)
{
	CK_BYTE hmac[SHA1_HASH_SIZE];
	SIGN_VERIFY_CONTEXT hmac_ctx;
	CK_ULONG hmac_len, len;
	CK_RV rc;

	if (!sess || !ctx || !in_data || !signature) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (ctx->mech.mechanism == CKM_SHA_1_HMAC_GENERAL)
		hmac_len = *(CK_ULONG *)ctx->mech.pParameter;
	else
		hmac_len = SHA1_HASH_SIZE;

	memset(&hmac_ctx, 0, sizeof(SIGN_VERIFY_CONTEXT));

	rc = sign_mgr_init(sess, &hmac_ctx, &ctx->mech, FALSE, ctx->key);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_SIGN_INIT);
		goto done;
	}
	len = sizeof(hmac);
	rc = sign_mgr_sign(sess, FALSE, &hmac_ctx, in_data, in_data_len,
			   hmac, &len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_SIGN);
		goto done;
	}
	if ((len != hmac_len) || (len != sig_len)) {
		OCK_LOG_ERR(ERR_SIGNATURE_LEN_RANGE);
		rc = CKR_SIGNATURE_LEN_RANGE;
		goto done;
	}

	if (memcmp(hmac, signature, hmac_len) != 0) {
		OCK_LOG_ERR(ERR_SIGNATURE_INVALID);
		rc = CKR_SIGNATURE_INVALID;
	}

done:
	sign_mgr_cleanup(&hmac_ctx);
	return rc;
}

CK_RV sha2_hmac_verify(SESSION *sess, SIGN_VERIFY_CONTEXT *ctx,
		       CK_BYTE *in_data, CK_ULONG in_data_len,
		       CK_BYTE *signature, CK_ULONG sig_len)
{
	CK_BYTE hmac[SHA2_HASH_SIZE];
	SIGN_VERIFY_CONTEXT hmac_ctx;
	CK_ULONG hmac_len, len;
	CK_RV rc;

	if (!sess || !ctx || !in_data || !signature) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}

	if (ctx->mech.mechanism == CKM_SHA256_HMAC_GENERAL)
		hmac_len = *(CK_ULONG *)ctx->mech.pParameter;
	else
		hmac_len = SHA2_HASH_SIZE;

	memset(&hmac_ctx, 0, sizeof(SIGN_VERIFY_CONTEXT));

	rc = sign_mgr_init(sess, &hmac_ctx, &ctx->mech, FALSE, ctx->key);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_SIGN_INIT);
		goto done;
	}

	len = sizeof(hmac);
	rc = sign_mgr_sign(sess, FALSE, &hmac_ctx, in_data, in_data_len,
			   hmac, &len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_SIGN);
		goto done;
	}

	if ((len != hmac_len) || (len != sig_len)) {
		OCK_LOG_ERR(ERR_SIGNATURE_LEN_RANGE);
		rc = CKR_SIGNATURE_LEN_RANGE;
		goto done;
	}

	if (memcmp(hmac, signature, hmac_len) != 0) {
		OCK_LOG_ERR(ERR_SIGNATURE_INVALID);
		rc = CKR_SIGNATURE_INVALID;
	}

done:
	sign_mgr_cleanup(&hmac_ctx);
	return rc;
}

CK_RV sha3_hmac_verify(SESSION *sess, SIGN_VERIFY_CONTEXT *ctx,
		       CK_BYTE *in_data, CK_ULONG in_data_len,
		       CK_BYTE *signature, CK_ULONG sig_len)
{
	CK_BYTE hmac[SHA3_HASH_SIZE];
	SIGN_VERIFY_CONTEXT hmac_ctx;
	CK_ULONG hmac_len, len;
	CK_RV rc;

	if (!sess || !ctx || !in_data || !signature) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}
	if (ctx->mech.mechanism == CKM_SHA384_HMAC_GENERAL)
		hmac_len = *(CK_ULONG *)ctx->mech.pParameter;
	else
		hmac_len = SHA3_HASH_SIZE;

	memset(&hmac_ctx, 0, sizeof(SIGN_VERIFY_CONTEXT));

	rc = sign_mgr_init(sess, &hmac_ctx, &ctx->mech, FALSE, ctx->key);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_SIGN_INIT);
		goto done;
	}
	len = sizeof(hmac);
	rc = sign_mgr_sign(sess, FALSE, &hmac_ctx, in_data, in_data_len,
			   hmac, &len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_SIGN);
		goto done;
	}
	if ((len != hmac_len) || (len != sig_len)) {
		OCK_LOG_ERR(ERR_SIGNATURE_LEN_RANGE);
		rc = CKR_SIGNATURE_LEN_RANGE;
		goto done;
	}

	if (memcmp(hmac, signature, hmac_len) != 0) {
		OCK_LOG_ERR(ERR_SIGNATURE_INVALID);
		rc = CKR_SIGNATURE_INVALID;
	}
done:
	sign_mgr_cleanup(&hmac_ctx);
	return rc;
}

CK_RV sha5_hmac_verify(SESSION *sess, SIGN_VERIFY_CONTEXT *ctx,
		       CK_BYTE *in_data, CK_ULONG in_data_len,
		       CK_BYTE *signature, CK_ULONG sig_len)
{
	CK_BYTE hmac[SHA5_HASH_SIZE];
	SIGN_VERIFY_CONTEXT hmac_ctx;
	CK_ULONG hmac_len, len;
	CK_RV rc;

	if (!sess || !ctx || !in_data || !signature) {
		OCK_LOG_ERR(ERR_FUNCTION_FAILED);
		return CKR_FUNCTION_FAILED;
	}
	if (ctx->mech.mechanism == CKM_SHA512_HMAC_GENERAL)
		hmac_len = *(CK_ULONG *)ctx->mech.pParameter;
	else
		hmac_len = SHA5_HASH_SIZE;

	memset(&hmac_ctx, 0, sizeof(SIGN_VERIFY_CONTEXT));

	rc = sign_mgr_init(sess, &hmac_ctx, &ctx->mech, FALSE, ctx->key);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_SIGN_INIT);
		goto done;
	}
	len = sizeof(hmac);
	rc = sign_mgr_sign(sess, FALSE, &hmac_ctx, in_data, in_data_len,
			   hmac, &len);
	if (rc != CKR_OK) {
		OCK_LOG_ERR(ERR_SIGN);
		goto done;
	}
	if ((len != hmac_len) || (len != sig_len)) {
		OCK_LOG_ERR(ERR_SIGNATURE_LEN_RANGE);
		rc = CKR_SIGNATURE_LEN_RANGE;
		goto done;
	}

	if (memcmp(hmac, signature, hmac_len) != 0) {
		OCK_LOG_ERR(ERR_SIGNATURE_INVALID);
		rc = CKR_SIGNATURE_INVALID;
	}
done:
	sign_mgr_cleanup(&hmac_ctx);
	return rc;
}

void sha1_init(DIGEST_CONTEXT *ctx)
{
	if (token_specific.t_sha_init == NULL)
		sw_sha1_init(ctx);
	else
		/* SAB XXX call token specific init... the init MUST
		 * allocate it's context
		 */
		token_specific.t_sha_init(ctx);
}

sha2_init(DIGEST_CONTEXT *ctx)
{
	if (token_specific.t_sha2_init == NULL) {
	        /* TODO: Software implementation here */
		return;
	 } else {
		/* SAB XXX call token specific init... the init
		 * MUST allocate it's context
		 */
		token_specific.t_sha2_init(ctx);
	}
}

sha3_init(DIGEST_CONTEXT *ctx)
{
	if (token_specific.t_sha3_init == NULL) {
		/* TODO: Software implementation here */
		return;
	} else {
		/* SAB XXX call token specific init... the init
		 * MUST allocate it's context
		 */
		token_specific.t_sha3_init(ctx);
	}
}

sha5_init(DIGEST_CONTEXT * ctx)
{
	if (token_specific.t_sha5_init == NULL) {
		/* TODO: Software implementation here */
		return;
	} else {
		/* SAB XXX call token specific init... the init
		 * MUST allocate it's context
		 */
		token_specific.t_sha5_init(ctx);
	}
}
