// --- BEGIN COPYRIGHT BLOCK ---
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; version 2 of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//
// (C) 2014 Red Hat, Inc.
// All rights reserved.
// --- END COPYRIGHT BLOCK ---

package org.dogtagpki.server.tps.mapping;

import java.util.HashMap;

import com.netscape.certsrv.base.EBaseException;
import com.netscape.certsrv.registry.IPluginInfo;
import com.netscape.cmscore.apps.EngineConfig;
import com.netscape.cmscore.base.ConfigStore;
import com.netscape.cmscore.registry.PluginRegistry;

/**
 * mappingResolverManager is a class for mapping resolver plugin
 * management
 *
 * @author cfu
 */
public class MappingResolverManager
{
    public static org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(MappingResolverManager.class);

    private static final String TOKEN_MAPPING_RESOLVER_TYPE = "tpsMappingResolver";
    public static final String PROP_RESOLVER_LIST = "list";
    public static final String PROP_RESOLVER_CLASS_ID = "class_id";

    protected static final String MAPPING_RESOLVER_CFG = "mappingResolver";
    protected PluginRegistry registry;
    protected HashMap<String, BaseMappingResolver> mappingResolvers = null;

    public MappingResolverManager() {
        mappingResolvers = new HashMap<>();
    }

    /**
     * initializes all mapping resolver plugin instances specified in
     * <instance-name>/conf/registry.cfg
     *
     * configuration e.g.
     *
     * registry.cfg:
     * types=tpsMappingResolver
     * tpsMappingResolver.ids=mappingTokenProfileResolverImpl
     * tpsMappingResolver.mappingTokenProfileResolverImpl.class=org.dogtagpki.server.tps.mapping.mappingResolver
     * tpsMappingResolver.mappingTokenProfileResolverImpl.desc=Mapping-based Token profile resolver
     * tpsMappingResolver.mappingTokenProfileResolverImpl.name=Mapping-based Token profile resolver
     *
     * CS.cfg :
     * registry.file=/var/lib/pki/pki-tomcat/conf/tps/registry.cfg
     * mappingResolver.list=formatMappingResolver,enrollMappingResolver,pinResetMappingResolver
     * mappingResolver.formatMappingResolver.class_id=mappingProfileResolverImpl
     * mappingResolver.formatMappingResolver.[plugin-specific configuration]
     *
     * op.format.mappingResolver=formatMappingResolver
     * ...
     * op.enroll.mappingResolver=enrollMappingResolver
     *
     * Note: "none" indicates no resolver plugin applied
     * op.format.mappingResolver=none
     */
    public void initMappingResolverInstances()
            throws EBaseException {

        String method = "mappingResolverManager.initMappingResolverInstance:";
        logger.debug(method + " begins");

        org.dogtagpki.server.tps.TPSEngine engine = org.dogtagpki.server.tps.TPSEngine.getInstance();

        registry = engine.getPluginRegistry();

        if (registry == null) {
            logger.warn(method + " registry null");
            return;
        }

        EngineConfig conf = engine.getConfig();
        ConfigStore prConf = conf.getSubStore(MAPPING_RESOLVER_CFG, ConfigStore.class);
        String profileList = prConf.getString(PROP_RESOLVER_LIST, "");

        for (String prInst : profileList.split(",")) {
            String classID = prConf.getString(prInst + "." + PROP_RESOLVER_CLASS_ID);
            logger.debug(method + " initializing classID=" + classID);
            IPluginInfo resolverInfo =
                    registry.getPluginInfo(TOKEN_MAPPING_RESOLVER_TYPE, classID);
            String resolverClass = resolverInfo.getClassName();
            BaseMappingResolver resolver = null;
            try {
                resolver = (BaseMappingResolver) Class.forName(resolverClass).getDeclaredConstructor().newInstance();
            } catch (Exception e) {
                logger.error(method + " resolver plugin Class.forName " + resolverClass + ": " + e.getMessage(), e);
                throw new EBaseException(e);
            }
            resolver.init(prInst);
            addResolver(prInst, resolver);
            logger.debug(method + " resolver instance added: " + prInst);
        }
    }

    public BaseMappingResolver getResolverInstance(String name) {
        return mappingResolvers.get(name);
    }

    public void addResolver(String instName, BaseMappingResolver resolver) {
        mappingResolvers.put(instName, resolver);
    }
}
