package generate

import (
	"testing"

	"github.com/containers/libpod/v2/pkg/domain/entities"
)

func TestValidateRestartPolicyPod(t *testing.T) {
	type podInfo struct {
		restart string
	}
	tests := []struct {
		name    string
		podInfo podInfo
		wantErr bool
	}{
		{"good-on", podInfo{restart: "no"}, false},
		{"good-on-success", podInfo{restart: "on-success"}, false},
		{"good-on-failure", podInfo{restart: "on-failure"}, false},
		{"good-on-abnormal", podInfo{restart: "on-abnormal"}, false},
		{"good-on-watchdog", podInfo{restart: "on-watchdog"}, false},
		{"good-on-abort", podInfo{restart: "on-abort"}, false},
		{"good-always", podInfo{restart: "always"}, false},
		{"fail", podInfo{restart: "foobar"}, true},
		{"failblank", podInfo{restart: ""}, true},
	}
	for _, tt := range tests {
		test := tt
		t.Run(tt.name, func(t *testing.T) {
			if err := validateRestartPolicy(test.podInfo.restart); (err != nil) != test.wantErr {
				t.Errorf("ValidateRestartPolicy() error = %v, wantErr %v", err, test.wantErr)
			}
		})
	}
}

func TestCreatePodSystemdUnit(t *testing.T) {
	podGood := `# pod-123abc.service
# autogenerated by Podman CI

[Unit]
Description=Podman pod-123abc.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target
Requires=container-1.service container-2.service
Before=container-1.service container-2.service

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=always
ExecStart=/usr/bin/podman start jadda-jadda-infra
ExecStop=/usr/bin/podman stop -t 10 jadda-jadda-infra
ExecStopPost=/usr/bin/podman stop -t 10 jadda-jadda-infra
PIDFile=/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid
KillMode=none
Type=forking

[Install]
WantedBy=multi-user.target default.target`

	podGoodNamedNew := `# pod-123abc.service
# autogenerated by Podman CI

[Unit]
Description=Podman pod-123abc.service
Documentation=man:podman-generate-systemd(1)
Wants=network.target
After=network-online.target
Requires=container-1.service container-2.service
Before=container-1.service container-2.service

[Service]
Environment=PODMAN_SYSTEMD_UNIT=%n
Restart=on-failure
ExecStartPre=/bin/rm -f %t/pod-123abc.pid %t/pod-123abc.pod-id
ExecStartPre=/usr/bin/podman pod create --infra-conmon-pidfile %t/pod-123abc.pid --pod-id-file %t/pod-123abc.pod-id --name foo --replace
ExecStart=/usr/bin/podman pod start --pod-id-file %t/pod-123abc.pod-id
ExecStop=/usr/bin/podman pod stop --ignore --pod-id-file %t/pod-123abc.pod-id -t 10
ExecStopPost=/usr/bin/podman pod rm --ignore -f --pod-id-file %t/pod-123abc.pod-id
PIDFile=%t/pod-123abc.pid
KillMode=none
Type=forking

[Install]
WantedBy=multi-user.target default.target`

	tests := []struct {
		name    string
		info    podInfo
		want    string
		new     bool
		wantErr bool
	}{
		{"pod",
			podInfo{
				Executable:       "/usr/bin/podman",
				ServiceName:      "pod-123abc",
				InfraNameOrID:    "jadda-jadda-infra",
				RestartPolicy:    "always",
				PIDFile:          "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:      10,
				PodmanVersion:    "CI",
				RequiredServices: []string{"container-1", "container-2"},
			},
			podGood,
			false,
			false,
		},
		{"pod --new",
			podInfo{
				Executable:       "/usr/bin/podman",
				ServiceName:      "pod-123abc",
				InfraNameOrID:    "jadda-jadda-infra",
				RestartPolicy:    "on-failure",
				PIDFile:          "/var/run/containers/storage/overlay-containers/639c53578af4d84b8800b4635fa4e680ee80fd67e0e6a2d4eea48d1e3230f401/userdata/conmon.pid",
				StopTimeout:      10,
				PodmanVersion:    "CI",
				RequiredServices: []string{"container-1", "container-2"},
				CreateCommand:    []string{"podman", "pod", "create", "--name", "foo"},
			},
			podGoodNamedNew,
			true,
			false,
		},
	}

	for _, tt := range tests {
		test := tt
		t.Run(tt.name, func(t *testing.T) {
			opts := entities.GenerateSystemdOptions{
				Files: false,
				New:   test.new,
			}
			got, err := executePodTemplate(&test.info, opts)
			if (err != nil) != test.wantErr {
				t.Errorf("CreatePodSystemdUnit() error = \n%v, wantErr \n%v", err, test.wantErr)
				return
			}
			if got != test.want {
				t.Errorf("CreatePodSystemdUnit() = \n%v\n---------> want\n%v", got, test.want)
			}
		})
	}
}
