#!/bin/bash



#Preupgrade assistant performs system upgradability assessment
#and gathers information required for successful operating system upgrade.
#Copyright (C) 2013 Red Hat Inc.
#Honza Horak <hhorak@redhat.com>
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.
. /usr/share/preupgrade/common.sh
check_applies_to "mysql-libs,mysql"
check_rpm_to "" ""
COMPONENT="mysql-libs,mysql"
#END GENERATED SECTION

# This check can be used if you need root privilegues
check_root

source ../mysql-common.sh

# filters comments that has # in the beggining of the line from stdin
filter_comments() {
    grep -v -e '^[[:space:]]*$' -e '^[[:space:]]*#' | sed 's|[[:space:]]*#.*||'
}

export OPTION_NOT_OK_FILE=option_not_ok
export OPTION_FIXED_FILE=option_fixed
rm -f $OPTION_NOT_OK_FILE
rm -f $OPTION_FIXED_FILE

# check options in file $1
# recursion level is in $2
check_options() {
    # recursion level limit so we do not cycle for ever
    [ $# -ne 2 ] && log_debug "check_options accepts 2 arguments" &&return 1
    [ $2 -gt 5 ] && log_debug "check_options recursion level exceeded" && return 2
    CONFIG_FILE=$1
    log_info  "checking config file $CONFIG_FILE"
    backup_config $CONFIG_FILE
    BACKUPED_CONFIG_FILE="${VALUE_TMP_PREUPGRADE}/$CONFIG_FILE"

    # remove comments and empty lines
    filtered_conf="`cat $CONFIG_FILE | filter_comments`"

    # starting from MariaDB/MySQL 5.5, innodb is the default storage engine,
    # so innodb plugin is not possible to be installed any more.
    echo "$filtered_conf" | grep -e 'plugin-load=innodb=' >/dev/null
    innodb_plugin_loaded=$?
    echo "$filtered_conf" | grep -e 'ignore-builtin-innodb' >/dev/null
    builtin_innodb_ignored=$?
    if [ $innodb_plugin_loaded -eq 0 ] && [ $builtin_innodb_ignored -eq 1 ]; then
        cat >> $SOLUTION_FILE <<EOF

* The InnoDB plugin is a default storage engine in MariaDB 5.5. You used InnoDB Plugin
and it was loaded using the plugin-load=innodb= configuration option, so you need to
remove this configuration option from configuration file $CONFIG_FILE as it does not
work in MariaDB 5.5 or use ignore-builtin-innodb.
EOF
        log_info "option 'plugin-load=innodb=' is not supported"
        sed -i -e 's/plugin-load=innodb=/#plugin-load=innodb=/g' $BACKUPED_CONFIG_FILE
        [ $? -eq 0 ] && touch $OPTION_FIXED_FILE
        touch $OPTION_NOT_OK_FILE
    fi

    # innodb_file_io_threads changed to innodb_read_io_threads and innodb_write_io_threads
    echo "$filtered_conf" | grep -e 'innodb_file_io_threads' >/dev/null
    if [ $? -eq 0 ]; then
        cat >> $SOLUTION_FILE <<EOF

* The configuration option innodb_file_io_threads was removed in MariaDB 5.5 
and new options innodb_read_io_threads and innodb_write_io_threads were
introduced instead. To ensure proper functionality, either remove the former
variable from the configuration file $CONFIG_FILE or replace it with the current variables.
EOF
        log_info "option 'innodb_file_io_threads' is not supported"
        sed -i -e 's/innodb_file_io_threads/#innodb_file_io_threads/g' $BACKUPED_CONFIG_FILE
        [ $? -eq 0 ] && touch $OPTION_FIXED_FILE
        touch $OPTION_NOT_OK_FILE
    fi

    # innodb_file_io_threads changed to innodb_read_io_threads and innodb_write_io_threads
    echo "$filtered_conf" | grep -e 'language\s*=' >/dev/null
    if [ $? -eq 0 ]; then
        cat >> $SOLUTION_FILE <<EOF

* MySQL 5.1 used the language variable for specifying the directory which
included the error message file. This option is now deprecated and has been
replaced by the lc_messages_dir and lc_messages options.
This also applies for configuration options in configuration file $CONFIG_FILE.
EOF
        log_info "option 'language' is deprecated"
        sed -i -e 's/language\s*=/#language=/g' $BACKUPED_CONFIG_FILE
        [ $? -eq 0 ] && touch $OPTION_FIXED_FILE
        touch $OPTION_NOT_OK_FILE
    fi

    obsolete_options_used=""
    for i in log-bin-trust-routine-creators table_lock_wait_timeout ; do
        echo "$filtered_conf" | grep $i >/dev/null
        if [ $? -eq 0 ]; then
            sed -i -e "s/${i}\s*/#${i}/g" $BACKUPED_CONFIG_FILE
            [ $? -eq 0 ] && touch $OPTION_FIXED_FILE
            log_info "option '$i' was removed in MariaDB 5.5"
            touch $OPTION_NOT_OK_FILE
            obsolete_options_used="${obsolete_options_used} '$i'"
        fi
    done

    obsolete_options_used=`echo $obsolete_options_used | sed 's|^ ||' | sed 's| |, |'`

    [ -n "$obsolete_options_used" ] && cat >> $SOLUTION_FILE <<EOF

* options $obsolete_options_used specified in your MySQL configuration file $CONFIG_FILE are not
  supported in MariaDB 5.5 in RHEL7 anymore.  Please remove them.
EOF

    # directive include includes the specified configuration file
    echo "$filtered_conf" | grep -uhe '!include[[:space:]]' \
                          | sed -e 's/.*!include[[:space:]]*\([^#]*\).*$/\1/' \
                          | while read includefile ; do 
        if [ -f "$includefile" ] ; then
            check_options "$includefile" $(($2 + 1)) || touch $OPTION_NOT_OK_FILE
        fi
    done

    # directive includedir includes all configuration files *.cnf
    echo "$filtered_conf" | grep -uhe '!includedir[[:space:]]' \
                          | sed -e 's/.*!include[[:space:]]*\([^#]*\).*$/\1/' \
                          | while read includedir ; do
        for includefile in ${includedir}/*.cnf ; do
            if [ -f "$includefile" ] ; then
                check_options "$includefile" $(($2 + 1)) || touch $OPTION_NOT_OK_FILE
            fi
        done
    done

}

# we don't check users' configurations, since they don't provide
# server configuration
check_options /etc/my.cnf 1

if [ ! -f $OPTION_NOT_OK_FILE ]; then
    log_info "No options problem in MySQL configuration found."
fi

if [ -f $OPTION_FIXED_FILE ] ; then
    log_info "Configuration files were fixed."
    exit $RESULT_FIXED
fi

exit $RESULT_INFORMATIONAL

