#
# (C) Copyright 2003-2006 Hewlett-Packard Development Company, L.P.
# (C) Copyright 2006-2007 Novell, Inc.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Author: Tim Potter <tpot@hp.com>
# Author: Bart Whiteley <bwhiteley@suse.de>
#

# TODO 2/16 KS: Sort this issue out and see if we need to fix. These two
#               issues are found repeatedly in this file so pylint disabled.
# pylint: disable=non-parent-init-called,super-init-not-called

"""
Functions to create XML documens and elements conforming to the DMTF
standard DSP0201, Representation of CIM in XML, v2.2.

  http://www.dmtf.org/standards/wbem/DSP201.html
  http://www.dmtf.org/standards/published_documents/DSP201.pdf

Elements generated by this module should conform to version 2.2 of the
DTD:

  http://www.dmtf.org/standards/wbem/CIM_DTD_V22.dtd

There should be one class for each element described in the DTD.  The
constructors take builtin Python types, or other cim_xml classes where
child elements are required.

Every class is a subclass of the Element class and so shares the same
attributes and methods, and can be used with the built-in Python XML
handling routines.  In particular you can call the toxml() and
toprettyxml() methods to generate XML.

Note that converting using toprettyxml() inserts whitespace which may
corrupt the data in the XML (!!) so you should only do this when
displaying to humans who can ignore it, and never for computers.  XML
always passes through all non-markup whitespace.

"""

from xml.dom.minidom import Element, Text, CDATASection
import six

__all__ = []

def _text(data):
    """Return a minidom text node with the specified data string.

    When converting that node to an XML string using ``toxml()``, XML-escaping
    will be applied if needed (for example, ``_text('&').toxml() = '&amp;'``).

    Note: The API for the minidom text node function has changed in
    Python 2.3.  The code for older Python versions has been removed from this
    function in PyWBEM 0.8; the Python version check is now done in
    __init__.py.
    """

    # The following initialization approach requires Python 2.3 or higher.
    t = Text()
    t.data = data

    return t

# Switch controlling whether the escaping of special XML characters in any
# request messages sent to the WBEM server will be done using CDATA
# sections (if True), or using XML entity references (e.g. &amp;) (if
# False).
# Using XML-based escaping generates a simpler CIM-XML payload.
# Standards-conforming WBEM servers need to support both ways.
# You only need to set this to True if the WBEM server has no support for
# XML-based escaping, but does have support for CDATA-based escaping.
_CDATA_ESCAPING = False

def _pcdata_nodes(pcdata):
    """Return a list of minidom nodes with the properly escaped ``pcdata``
    inside.

    The following special XML characters are escaped:
      * left angle bracket (<)
      * Right angle bracket (>)
      * Ampersand (&)

    By default, XML-based escaping is used for these characters. XML-based
    escaping will cause the corresponding XML entity references to be used
    (for example, the ampersand character ``&`` will be represented as
    ``&amp;``, and the returned list contains one text node with the escaped
    pcdata string.

    Nesting of escaped pcdata is naturally supported with XML-based escaping.
    For example, if the pcdata string is ``a&amp;b``, the XML-escaped string
    will be ``a&amp;amp;b``.

    If the ``cim_xml._CDATA_ESCAPING`` switch is set to True, CDATA-based
    escaping is used instead. CDATA-based escaping will cause a CDATA section
    to be used for the entire string, or consecutive CDATA sequences (see
    discussion of nesting, below). The returned node list contains only CDATA
    section nodes. Example: The pcdata string ``a<b>c`` will become
    ``<![CDATA[a<b>]]>``, allowing the special XML characters to be used
    unchanged inside of the CDATA section.

    Nesting of escaped pcdata is supported with CDATA-based escaping, by using
    the following approach: If the input pcdata string already contains CDATA
    sections, they are split into separate strings, splitting the CDATA end
    token string in the middle, and these part strings are CDATA-escaped
    separately. See http://en.wikipedia.org/wiki/CDATA#Nesting for details.

    Escaping of already escaped pcdata is needed in support of nested embedded
    instances. That requires that each level of escaping can lateron be
    unescaped, one at a time.
    """

    nodelist = []

    if _CDATA_ESCAPING and isinstance(pcdata, six.string_types) and \
       (pcdata.find("<") >= 0 or \
        pcdata.find(">") >= 0 or \
        pcdata.find("&") >= 0):

        # In order to support nesting of CDATA sections, we represent pcdata
        # that already contains CDATA sections by multiple new CDATA sections
        # whose boundaries split the end marker of the already existing CDATA
        # sections.

        pcdata_part_list = pcdata.split("]]>")
        # ']]>' is the complete CDATA section end marker

        i = 0
        for pcdata_part in pcdata_part_list:
            i += 1

            left = "" if i == 1 else "]>"
            # ']>' is right part of CDATA section end marker

            right = "" if i == len(pcdata_part_list) else "]"
            # "]" is left part of CDATA section end marker

            # The following initialization approach requires Python 2.3 or
            # higher.
            node = CDATASection()
            node.data = left + pcdata_part + right

            nodelist.append(node)

    else:
        # The following automatically uses XML entity references
        # for escaping.
        # pylint: disable=redefined-variable-type
        node = _text(pcdata)

        nodelist.append(node)

    return nodelist

class CIMElement(Element):
    """A base class that has a few bonus helper methods."""

    ownerDocument = None

    def setName(self, name):
        """Set the NAME attribute of the element."""
        self.setAttribute('NAME', name)

    def setOptionalAttribute(self, name, value):
        """Set an attribute if the value parameter is not None."""
        if value is not None:
            self.setAttribute(name, value)

    def appendOptionalChild(self, child):
        """Append a child element which can be None."""
        if child is not None:
            self.appendChild(child)

    def appendChildren(self, children):
        """Append a list or tuple of children."""
        for child in children:
            self.appendChild(child)

# Root element

class CIM(CIMElement):
    """
    The CIM element is the root element of every XML Document that is
    valid with respect to this schema.

    Each document takes one of two forms; it either contains a single
    MESSAGE element defining a CIM message (to be used in the HTTP
    mapping), or it contains a DECLARATION element used to declare a
    set of CIM objects.

    <!ELEMENT CIM (MESSAGE | DECLARATION)>
    <!ATTLIST CIM
        CIMVERSION CDATA #REQUIRED
        DTDVERSION CDATA #REQUIRED>
    """

    def __init__(self, data, cim_version, dtd_version):
        Element.__init__(self, 'CIM')
        self.setAttribute('CIMVERSION', cim_version)
        self.setAttribute('DTDVERSION', dtd_version)
        self.appendChild(data)

# Object declaration elements

class DECLARATION(CIMElement):
    """
    The DECLARATION element defines a set of one or more declarations
    of CIM objects.  These are partitioned into logical declaration
    subsets.

    <!ELEMENT DECLARATION (DECLGROUP | DECLGROUP.WITHNAME |
                           DECLGROUP.WITHPATH)+>
    """

    def __init__(self, data):
        Element.__init__(self, 'DECLARATION')
        self.appendChildren(data)

class DECLGROUP(CIMElement):
    """
    The DECLGROUP element defines a logical set of CIM Class, Instance
    and Qualifier declarations.  It MAY optionally include a
    NAMESPACEPATH or LOCALNAMESPACEPATH element which, if present,
    defines the common namespace in which all objects within the group
    are declared.

    <!ELEMENT DECLGROUP ((LOCALNAMESPACEPATH | NAMESPACEPATH)?,
                         QUALIFIER.DECLARATION*, VALUE.OBJECT*)>
    """

    def __init__(self, data):
        Element.__init__(self, 'DECLGROUP')
        self.appendChildren(data)

class DECLGROUP_WITHNAME(CIMElement):
    # pylint: disable=invalid-name
    """
    The DECLGROUP.WITHNAME element defines a logical set of CIM Class,
    Instance and Qualifier declarations.  It MAY optionally include a
    NAMESPACEPATH or LOCALNAMESPACEPATH element which, if present,
    defines the common namespace in which all objects within the group
    are declared.

    <!ELEMENT DECLGROUP.WITHNAME ((LOCALNAMESPACEPATH | NAMESPACEPATH)?,
                                   QUALIFIER.DECLARATION*, VALUE.NAMEDOBJECT*)>
    """

    def __init__(self, data):
        Element.__init__(self, 'DECLGROUP.WITHNAME')
        self.appendChildren(data)

class DECLGROUP_WITHPATH(CIMElement):
    # pylint: disable=invalid-name
    """
    The DECLGROUP.WITHPATH element defines a logical set of CIM Class
    and Instance declarations.  Each object is declared with its own
    independent naming and location information.

    <!ELEMENT DECLGROUP.WITHPATH  (VALUE.OBJECTWITHPATH |
                                   VALUE.OBJECTWITHLOCALPATH)*>
    """

    def __init__(self, data):
        Element.__init__(self, 'DECLGROUP.WITHPATH')
        self.appendChildren(data)

class QUALIFIER_DECLARATION(CIMElement):
    # pylint: disable=invalid-name
    """
    The QUALIFIER.DECLARATION element defines a single CIM Qualifier
    declaration.

    <!ELEMENT QUALIFIER.DECLARATION (SCOPE?, (VALUE | VALUE.ARRAY)?)>
    <!ATTLIST QUALIFIER.DECLARATION
         %CIMName;
         %CIMType;               #REQUIRED
         ISARRAY    (true|false) #IMPLIED
         %ArraySize;
         %QualifierFlavor;>
    """

    def __init__(self, name, type_, value, is_array=None,
                 array_size=None, qualifier_scopes={},
                 overridable=None, tosubclass=None,
                 toinstance=None, translatable=None):

        Element.__init__(self, 'QUALIFIER.DECLARATION')
        self.setName(name)
        self.setAttribute('TYPE', type_)
        if is_array is not None:
            self.setOptionalAttribute('ISARRAY', str(is_array).lower())
        if array_size is not None:
            self.setAttribute('ARRAYSIZE', str(array_size))
        if overridable is not None:
            self.setAttribute('OVERRIDABLE', str(overridable).lower())
        if tosubclass is not None:
            self.setAttribute('TOSUBCLASS', str(tosubclass).lower())
        if toinstance is not None:
            self.setAttribute('TOINSTANCE', str(toinstance).lower())
        if translatable is not None:
            self.setAttribute('TRANSLATABLE', str(translatable).lower())

        if qualifier_scopes:
            scope = SCOPE(qualifier_scopes)
            self.appendOptionalChild(scope)
        if value is not None:
            if is_array:
                xval = VALUE_ARRAY(value)
            else:
                xval = VALUE(value)
            self.appendOptionalChild(xval)

class SCOPE(CIMElement):
    """
    The SCOPE element defines the scope of a QUALIFIER.DECLARATION in
    the case that there are restrictions on the scope of the Qualifier
    declaration.

    <!ELEMENT SCOPE EMPTY>
    <!ATTLIST SCOPE
         CLASS        (true|false)      'false'
         ASSOCIATION  (true|false)      'false'
         REFERENCE    (true|false)      'false'
         PROPERTY     (true|false)      'false'
         METHOD       (true|false)      'false'
         PARAMETER    (true|false)      'false'
         INDICATION   (true|false)      'false'>
    """

    def __init__(self, scopes={}):
        Element.__init__(self, 'SCOPE')
        if 'any' in scopes and scopes['any']:
            scopes = {'CLASS': True,
                      'ASSOCIATION': True,
                      'REFERENCE': True,
                      'PROPERTY': True,
                      'METHOD': True,
                      'PARAMETER': True,
                      'INDICATION': True}
        for k, v in scopes.items():
            self.setOptionalAttribute(k, str(v).lower())

# Object value elements

class VALUE(CIMElement):
    """
    The VALUE element is used to define a single (non-array and
    non-reference) CIM Property value, CIM Qualifier value, or a CIM
    Method Parameter value.

    <!ELEMENT VALUE (#PCDATA)>
    """

    def __init__(self, pcdata):
        Element.__init__(self, 'VALUE')
        if pcdata is not None:
            self.appendChildren(_pcdata_nodes(pcdata))

class VALUE_ARRAY(CIMElement):
    # pylint: disable=invalid-name
    """
    The VALUE.ARRAY element is used to represent the value of a CIM
    Property or Qualifier that has an array type.

    <!ELEMENT VALUE.ARRAY (VALUE*)>
    """

    def __init__(self, values):
        Element.__init__(self, 'VALUE.ARRAY')
        self.appendChildren(values)

class VALUE_REFERENCE(CIMElement):
    # pylint: disable=invalid-name
    """
    The VALUE.REFERENCE element is used to define a single CIM
    reference Property value.

    <!ELEMENT VALUE.REFERENCE (CLASSPATH | LOCALCLASSPATH | CLASSNAME |
                               INSTANCEPATH | LOCALINSTANCEPATH |
                               INSTANCENAME)>
    """

    def __init__(self, data):
        Element.__init__(self, 'VALUE.REFERENCE')
        self.appendChild(data)

class VALUE_REFARRAY(CIMElement):
    # pylint: disable=invalid-name
    """
    The VALUE.REFARRAY element is used to represent the value of an
    array of CIM references.

    <!ELEMENT VALUE.REFARRAY (VALUE.REFERENCE*)>
    """

    def __init__(self, data):
        Element.__init__(self, 'VALUE.REFARRAY')
        self.appendChildren(data)

class VALUE_OBJECT(CIMElement):
    # pylint: disable=invalid-name
    """
    The VALUE.OBJECT element is used to define a value which is
    comprised of a single CIM Class or Instance definition.

    <!ELEMENT VALUE.OBJECT (CLASS | INSTANCE)>
    """

    def __init__(self, data):
        Element.__init__(self, 'VALUE.OBJECT')
        self.appendChild(data)

class VALUE_NAMEDINSTANCE(CIMElement):
    # pylint: disable=invalid-name
    """
    The VALUE.NAMEDINSTANCE element is used to define a value which
    is comprised of a single named CIM Instance definition.

    <!ELEMENT VALUE.NAMEDINSTANCE (INSTANCENAME, INSTANCE)>
    """

    def __init__(self, instancename, instance):
        Element.__init__(self, 'VALUE.NAMEDINSTANCE')
        self.appendChild(instancename)
        self.appendChild(instance)

class VALUE_NAMEDOBJECT(CIMElement):
    # pylint: disable=invalid-name
    """
    The VALUE.NAMEDOBJECT element is used to define a value which
    is comprised of a single named CIM Class or Instance definition.

    <!ELEMENT VALUE.NAMEDOBJECT (CLASS | (INSTANCENAME, INSTANCE))>
    """

    def __init__(self, data):
        Element.__init__(self, 'VALUE.NAMEDOBJECT')
        if isinstance(data, (tuple, list)):
            self.appendChildren(data)
        else:
            self.appendChild(data)

class VALUE_OBJECTWITHLOCALPATH(CIMElement):
    # pylint: disable=invalid-name
    """
    The VALUE.OBJECTWITHLOCALPATH element is used to define a value
    which is comprised of a single CIM Object (Class or Instance)
    definition with additional information that defines the local path
    to that Object.

    <!ELEMENT VALUE.OBJECTWITHLOCALPATH ((LOCALCLASSPATH, CLASS) |
                                         (LOCALINSTANCEPATH, INSTANCE))>
    """

    def __init__(self, data1, data2):
        Element.__init__(self, 'VALUE.OBJECTWITHLOCALPATH')
        self.appendChild(data1)
        self.appendChild(data2)

class VALUE_OBJECTWITHPATH(CIMElement):
    # pylint: disable=invalid-name
    """
    The VALUE.OBJECTWITHPATH element is used to define a value
    which is comprised of a single CIM Object (Class or Instance)
    definition with additional information that defines the absolute
    path to that Object.

    <!ELEMENT VALUE.OBJECTWITHPATH ((CLASSPATH, CLASS) |
                                    (INSTANCEPATH, INSTANCE))>
    """

    def __init__(self, data1, data2):
        Element.__init__(self, 'VALUE.OBJECTWITHPATH')
        self.appendChild(data1)
        self.appendChild(data2)

class VALUE_NULL(CIMElement):
    # pylint: disable=invalid-name
    """
    The VALUE.NULL element is used to represent a TABLECELL that has
    no assigned value.

    <!ELEMENT VALUE.NULL EMPTY>
    """

    def __init__(self):
        Element.__init__(self, 'VALUE.NULL')

# Object naming and location elements

class NAMESPACEPATH(CIMElement):
    # pylint: disable=invalid-name
    """
    The NAMESPACEPATH element is used to define a Namespace Path. It
    consists of a HOST element and a LOCALNAMESPACE element.

    <!ELEMENT NAMESPACEPATH (HOST, LOCALNAMESPACEPATH)>
    """

    def __init__(self, host, localnamespacepath):
        Element.__init__(self, 'NAMESPACEPATH')
        self.appendChild(host)
        self.appendChild(localnamespacepath)

class LOCALNAMESPACEPATH(CIMElement):
    # pylint: disable=invalid-name
    """
    The LOCALNAMESPACEPATH element is used to define a local Namespace
    path (one without a Host component). It consists of one or more
    NAMESPACE elements (one for each namespace in the path).

    <!ELEMENT LOCALNAMESPACEPATH (NAMESPACE+)>
    """

    def __init__(self, namespaces):
        Element.__init__(self, 'LOCALNAMESPACEPATH')
        self.appendChildren(namespaces)

class HOST(CIMElement):
    # pylint: disable=invalid-name
    """
    The HOST element is used to define a single Host. The element
    content MUST specify a legal value for a hostname in accordance
    with the CIM specification.

    <!ELEMENT HOST (#PCDATA)>
    """

    def __init__(self, pcdata):
        Element.__init__(self, 'HOST')
        self.appendChild(_text(pcdata))

class NAMESPACE(CIMElement):
    # pylint: disable=invalid-name
    """
    The NAMESPACE element is used to define a single Namespace
    component of a Namespace path.

    <!ELEMENT NAMESPACE EMPTY>
    <!ATTLIST NAMESPACE
        %CIMName;>
    """

    def __init__(self, name):
        Element.__init__(self, 'NAMESPACE')
        self.setName(name)

class CLASSPATH(CIMElement):
    # pylint: disable=invalid-name
    """
    The CLASSPATH element defines the absolute path to a CIM Class. It
    is formed from a namespace path and Class name.

    <!ELEMENT CLASSPATH (NAMESPACEPATH, CLASSNAME)>
    """

    def __init__(self, namespacepath, classname):
        Element.__init__(self, 'CLASSPATH')
        self.appendChild(namespacepath)
        self.appendChild(classname)


class LOCALCLASSPATH(CIMElement):
    # pylint: disable=invalid-name
    """
    The LOCALCLASSPATH element defines the a local path to a CIM
    Class. It is formed from a local namespace path and Class name.

    <!ELEMENT LOCALCLASSPATH (LOCALNAMESPACEPATH, CLASSNAME)>
    """

    def __init__(self, localnamespacepath, classname):
        Element.__init__(self, 'LOCALCLASSPATH')
        self.appendChild(localnamespacepath)
        self.appendChild(classname)

class CLASSNAME(CIMElement):
    # pylint: disable=invalid-name
    """
    The CLASSNAME element defines the qualifying name of a CIM Class.

    <!ELEMENT CLASSNAME EMPTY>
    <!ATTLIST CLASSNAME
        %CIMName;>
    """

    def __init__(self, classname):
        Element.__init__(self, 'CLASSNAME')
        self.setName(classname)

class INSTANCEPATH(CIMElement):
    # pylint: disable=invalid-name
    """
    The INSTANCEPATH element defines the absolute path to a CIM
    Instance. It is comprised of a Namespace path and an Instance Name
    (model path).

    <!ELEMENT INSTANCEPATH (NAMESPACEPATH, INSTANCENAME)>
    """

    def __init__(self, namespacepath, instancename):
        Element.__init__(self, 'INSTANCEPATH')
        self.appendChild(namespacepath)
        self.appendChild(instancename)

class LOCALINSTANCEPATH(CIMElement):
    # pylint: disable=invalid-name
    """
    The LOCALINSTANCEPATH element defines the local path to a CIM
    Instance. It is comprised of a local Namespace path and an
    Instance Name (model path).

    <!ELEMENT LOCALINSTANCEPATH (LOCALNAMESPACEPATH, INSTANCENAME)>
    """

    def __init__(self, localpath, instancename):
        Element.__init__(self, 'LOCALINSTANCEPATH')
        self.appendChild(localpath)
        self.appendChild(instancename)

class INSTANCENAME(CIMElement):
    # pylint: disable=invalid-name
    """
    The INSTANCENAME element defines the location of a CIM Instance
    within a Namespace (it is referred to in the CIM Specification
    as a Model Path). It is comprised of a class name and a key
    binding information.

    If the Class has a single key property, then a single KEYVALUE or
    VALUE.REFERENCE subelement may be used to describe the
    (necessarily) unique key value without a key name. Alternatively a
    single KEYBINDING subelement may be used instead.

    If the Class has more than one key property, then a KEYBINDING
    subelement MUST appear for each key.

    If there are no key-bindings specified, the instance is assumed to
    be a singleton instance of a keyless Class.

    <!ELEMENT INSTANCENAME (KEYBINDING* | KEYVALUE? | VALUE.REFERENCE?)>
    <!ATTLIST INSTANCENAME
        %ClassName;>
    """

    def __init__(self, classname, data):
        Element.__init__(self, 'INSTANCENAME')
        self.setAttribute('CLASSNAME', classname)
        if data is not None:
            if isinstance(data, list):
                self.appendChildren(data)
            else:
                self.appendChild(data)

class OBJECTPATH(CIMElement):
    # pylint: disable=invalid-name
    """
    The OBJECTPATH element is used to define a full path to a single
    CIM Object (Class or Instance).

    <!ELEMENT OBJECTPATH (INSTANCEPATH | CLASSPATH)>
    """

    def __init__(self, data):
        Element.__init__(self, 'OBJECTPATH')
        self.appendChild(data)

class KEYBINDING(CIMElement):
    # pylint: disable=invalid-name
    """
    The KEYBINDING element defines a single key property value binding.

    <!ELEMENT KEYBINDING (KEYVALUE | VALUE.REFERENCE)>
    <!ATTLIST KEYBINDING
        %CIMName;>
    """

    def __init__(self, name, data):
        Element.__init__(self, 'KEYBINDING')
        self.setName(name)
        self.appendChild(data)

class KEYVALUE(CIMElement):
    # pylint: disable=invalid-name
    """
    The KEYVALUE element defines a single property key value when the
    key property is a non-reference type.

    <!ELEMENT KEYVALUE (#PCDATA)>
    <!ATTLIST KEYVALUE
        VALUETYPE    (string|boolean|numeric)  'string'
        %CIMType;    #IMPLIED>
    """

    def __init__(self, data, value_type=None, cim_type=None):

        Element.__init__(self, 'KEYVALUE')

        if value_type is None:
            self.setAttribute('VALUETYPE', 'string')
        else:
            self.setAttribute('VALUETYPE', value_type)

        self.setOptionalAttribute('TYPE', cim_type)

        if data != None:
            self.appendChild(_text(data))

# Object definition elements

class CLASS(CIMElement):
    # pylint: disable=invalid-name
    """
    The CLASS element defines a single CIM Class.

    <!ELEMENT CLASS (QUALIFIER*, (PROPERTY | PROPERTY.ARRAY |
                     PROPERTY.REFERENCE)*, METHOD*)>
    <!ATTLIST CLASS
        %CIMName;
        %SuperClass;>
    """

    def __init__(self, classname, properties=[], methods=[],
                 qualifiers=[], superclass=None):
        Element.__init__(self, 'CLASS')
        self.setName(classname)
        self.setOptionalAttribute('SUPERCLASS', superclass)
        self.appendChildren(qualifiers + properties + methods)

class INSTANCE(CIMElement):
    # pylint: disable=invalid-name
    """
    The INSTANCE element defines a single CIM Instance of a CIM Class.

    <!ELEMENT INSTANCE (QUALIFIER*, (PROPERTY | PROPERTY.ARRAY |
                                     PROPERTY.REFERENCE)*)>
    <!ATTLIST INSTANCE
         %ClassName;
         xml:lang   NMTOKEN  #IMPLIED>
    """
    def __init__(self, classname, properties=[], qualifiers=[],
                 xml_lang=None):
        Element.__init__(self, 'INSTANCE')
        self.setAttribute('CLASSNAME', classname)
        self.setOptionalAttribute('xml:lang', xml_lang)
        self.appendChildren(qualifiers + properties)

class QUALIFIER(CIMElement):
    # pylint: disable=invalid-name
    """
    The QUALIFIER element defines a single CIM Qualifier. If the
    Qualifier has a non-array type, it contains a single VALUE element
    representing the value of the Qualifier. If the Qualifier has an
    array type, it contains a single VALUE.ARRAY element to represent
    the value.

    If the Qualifier has no assigned value then the VALUE element MUST
    be absent.

    <!ELEMENT QUALIFIER ((VALUE | VALUE.ARRAY)?)>
    <!ATTLIST QUALIFIER
        %CIMName;
        %CIMType;               #REQUIRED
        %Propagated;
        %QualifierFlavor;
        xml:lang  NMTOKEN  #IMPLIED>
    """

    def __init__(self, name, type_, value=None, propagated=None,
                 overridable=None, tosubclass=None, toinstance=None,
                 translatable=None, xml_lang=None):

        Element.__init__(self, 'QUALIFIER')

        self.setName(name)
        self.setAttribute('TYPE', type_)

        if propagated is not None:
            self.setAttribute('PROPAGATED', str(propagated).lower())

        if overridable is not None:
            self.setAttribute('OVERRIDABLE', str(overridable).lower())
        if tosubclass is not None:
            self.setAttribute('TOSUBCLASS', str(tosubclass).lower())
        if toinstance is not None:
            self.setAttribute('TOINSTANCE', str(toinstance).lower())
        if translatable is not None:
            self.setAttribute('TRANSLATABLE', str(translatable).lower())

        self.setOptionalAttribute('xml:lang', xml_lang)

        self.appendOptionalChild(value)

class PROPERTY(CIMElement):
    # pylint: disable=invalid-name
    """
    The PROPERTY element defines a single (non-array) CIM Property
    that is not a reference. It contains a single VALUE element
    representing the value of the Property.

    If the Property has no assigned value then the VALUE element MUST be
    absent.

    CIM Reference Properties are described using the
    PROPERTY.REFERENCE element.

    <!ELEMENT PROPERTY (QUALIFIER*, VALUE?)>
    <!ATTLIST PROPERTY
        %CIMName;
        %ClassOrigin;
        %Propagated;
        %CIMType;           #REQUIRED
        xml:lang   NMTOKEN  #IMPLIED>
    """

    def __init__(self, name, type_, value=None, class_origin=None,
                 propagated=None, qualifiers=[], xml_lang=None,
                 embedded_object=None):

        Element.__init__(self, 'PROPERTY')

        self.setName(name)
        self.setAttribute('TYPE', type_)

        self.setOptionalAttribute('CLASSORIGIN', class_origin)

        if propagated is not None:
            self.setAttribute('PROPAGATED', str(propagated).lower())

        self.setOptionalAttribute('xml:lang', xml_lang)

        self.setOptionalAttribute('EmbeddedObject', embedded_object)
        # Note on EmbeddedObject:
        # The CIM-XML standard requires the mixed case form, 'EmbeddedObject'.
        # Given that all other attributes are in upper case, this is an
        # inconsistency in the standard that cannot be fixed for backwards
        # compatibility reasons.
        # OpenPegasus 2.6 supported only the upper case form 'EMBEDDEDOBJECT'.
        # Later OpenPegasus versions fixed that by using 'EmbeddedObject' in
        # any responses and by accepting both 'EMBEDDEDOBJECT' and
        # 'EmbeddedObject' in any requests.
        # Adding the 'EMBEDDEDOBJECT' attribute to any requests (that is, in
        # this code here) in order to support the old OpenPegasus 2.6 does not
        # work unfortunately, because SFCB supports either form but not both
        # attributes in the same request.
        # As a result, the best choice is to use only the standards-conforming
        # mixed case form in any requests sent by PyWBEM.

        self.appendChildren(qualifiers)
        self.appendOptionalChild(value)

class PROPERTY_ARRAY(CIMElement):
    # pylint: disable=invalid-name
    """
    The PROPERTY.ARRAY element defines a single CIM Property with an
    array type. It contains a single VALUE.ARRAY element  representing
    the value of the Property.

    If the Property has no assigned value then the VALUE.ARRAY element
    MUST be absent.

    There is no element to model a Property that contains an array of
    references as this is not a valid Property type according to CIM.

    <!ELEMENT PROPERTY.ARRAY (QUALIFIER*, VALUE.ARRAY?)>
    <!ATTLIST PROPERTY.ARRAY
       %CIMName;
       %CIMType;           #REQUIRED
       %ArraySize;
       %ClassOrigin;
       %Propagated;
       xml:lang   NMTOKEN  #IMPLIED>
    """

    def __init__(self, name, type_, value_array=None, array_size=None,
                 class_origin=None, propagated=None, qualifiers=[],
                 xml_lang=None, embedded_object=None):

        Element.__init__(self, 'PROPERTY.ARRAY')

        self.setName(name)
        self.setAttribute('TYPE', type_)

        if array_size is not None:
            self.setAttribute('ARRAYSIZE', str(array_size))
        self.setOptionalAttribute('CLASSORIGIN', class_origin)
        self.setOptionalAttribute('EmbeddedObject', embedded_object)
        # See the note on EmbeddedObject in PROPERTY().

        if propagated is not None:
            self.setAttribute('PROPAGATED', str(propagated).lower())

        self.setOptionalAttribute('xml:lang', xml_lang)

        self.appendChildren(qualifiers)
        self.appendOptionalChild(value_array)

class PROPERTY_REFERENCE(CIMElement):
    # pylint: disable=invalid-name
    """
    The PROPERTY.REFERENCE element models a single CIM Property with
    reference semantics. In future the features of XML Linking may
    be used to identify linking elements within the XML Document; as
    XML Linking is currently only at Working Draft status no explicit
    dependencies have been made at this point.

    <!ELEMENT PROPERTY.REFERENCE (QUALIFIER*, VALUE.REFERENCE?)>
    <!ATTLIST PROPERTY.REFERENCE
        %CIMName;
        %ReferenceClass;
        %ClassOrigin;
        %Propagated;>
    """

    def __init__(self, name, value_reference=None, reference_class=None,
                 class_origin=None, propagated=None, qualifiers=[]):

        Element.__init__(self, 'PROPERTY.REFERENCE')

        self.setName(name)

        self.setOptionalAttribute('REFERENCECLASS', reference_class)
        self.setOptionalAttribute('CLASSORIGIN', class_origin)

        if propagated is not None:
            self.setAttribute('PROPAGATED', str(propagated).lower())

        self.appendChildren(qualifiers)
        self.appendOptionalChild(value_reference)

class METHOD(CIMElement):
    # pylint: disable=invalid-name
    """
    The METHOD element defines a single CIM Method. It may have
    Qualifiers, and zero or more parameters.

    The order of the PARAMETER, PARAMETER.REFERENCE, PARAMETER.ARRAY
    and PARAMETER.REFARRAY subelements is not significant.

    <!ELEMENT METHOD (QUALIFIER*, (PARAMETER | PARAMETER.REFERENCE |
                                   PARAMETER.ARRAY | PARAMETER.REFARRAY)*)>
    <!ATTLIST METHOD
        %CIMName;
        %CIMType;          #IMPLIED
        %ClassOrigin;
        %Propagated;>
    """

    def __init__(self, name, parameters=[], return_type=None,
                 class_origin=None, propagated=None, qualifiers=[]):

        Element.__init__(self, 'METHOD')

        self.setName(name)

        self.setOptionalAttribute('TYPE', return_type)
        self.setOptionalAttribute('CLASSORIGIN', class_origin)

        if propagated is not None:
            self.setAttribute('PROPAGATED', str(propagated).lower())

        self.appendChildren(qualifiers + parameters)

class PARAMETER(CIMElement):
    # pylint: disable=invalid-name
    """
    The PARAMETER element defines a single (non-array, non-reference)
    Parameter to a CIM Method. The parameter MAY have zero or more
    Qualifiers.

    <!ELEMENT PARAMETER (QUALIFIER*)>
    <!ATTLIST PARAMETER
        %CIMName;
        %CIMType;      #REQUIRED>
    """

    def __init__(self, name, type_, qualifiers=[]):
        Element.__init__(self, 'PARAMETER')
        self.setName(name)
        self.setAttribute('TYPE', type_)
        self.appendChildren(qualifiers)

class PARAMETER_REFERENCE(CIMElement):
    # pylint: disable=invalid-name
    """
    The PARAMETER.REFERENCE element defines a single reference
    Parameter to a CIM Method. The parameter MAY have zero or more
    Qualifiers.

    <!ELEMENT PARAMETER.REFERENCE (QUALIFIER*)>
    <!ATTLIST PARAMETER.REFERENCE
        %CIMName;
        %ReferenceClass;>
    """

    def __init__(self, name, reference_class=None, qualifiers=[]):
        Element.__init__(self, 'PARAMETER.REFERENCE')
        self.setName(name)
        self.setOptionalAttribute('REFERENCECLASS', reference_class)
        self.appendChildren(qualifiers)

class PARAMETER_ARRAY(CIMElement):
    # pylint: disable=invalid-name
    """
    The PARAMETER.ARRAY element defines a single Parameter to a CIM
    Method that has an array type. The parameter MAY have zero or more
    Qualifiers.

    <!ELEMENT PARAMETER.ARRAY (QUALIFIER*)>
    <!ATTLIST PARAMETER.ARRAY
        %CIMName;
        %CIMType;           #REQUIRED
        %ArraySize;>
    """

    def __init__(self, name, type_, array_size=None, qualifiers=[]):
        Element.__init__(self, 'PARAMETER.ARRAY')
        self.setName(name)
        self.setAttribute('TYPE', type_)
        if array_size is not None:
            self.setAttribute('ARRAYSIZE', str(array_size))
        self.appendChildren(qualifiers)

class PARAMETER_REFARRAY(CIMElement):
    # pylint: disable=invalid-name
    """
    The PARAMETER.REFARRAY element defines a single Parameter to a CIM
    Method that has an array of references type. The parameter MAY
    have zero or more Qualifiers.

    <!ELEMENT PARAMETER.REFARRAY (QUALIFIER*)>
    <!ATTLIST PARAMETER.REFARRAY
        %CIMName;
        %ReferenceClass;
        %ArraySize;>
    """

    def __init__(self, name, reference_class=None, array_size=None,
                 qualifiers=[]):
        Element.__init__(self, 'PARAMETER.REFARRAY')
        self.setName(name)
        self.setOptionalAttribute('REFERENCECLASS', reference_class)
        if array_size is not None:
            self.setAttribute('ARRAYSIZE', str(array_size))
        self.appendChildren(qualifiers)

class TABLECELL_DECLARATION(CIMElement): #pylint: disable=invalid-name
    # pylint: disable=invalid-name
    """
    The TABLECELL.DECLARATION element describes a TABLECELL that is
    not a reference or an array of references.


    <!ELEMENT TABLECELL.DECLARATION EMPTY>
    <!ATTLIST TABLECELL.DECLARATION
        %CIMName;
        %CIMType;                    #REQUIRED
        ISARRAY         (true|false) "false"
        %ARRAYSIZE;
        CELLPOS          CDATA       #REQUIRED
        SORTPOS          CDATA       #IMPLIED
        SORTDIR        (ASC|DESC)    #IMPLIED>
    """

class TABLECELL_REFERENCE(CIMElement):
    # pylint: disable=invalid-name
    """

    The TABLECELL.REFERENCE element defines a TABLECELL that contains
    reference or reference array values.

    <!ELEMENT TABLECELL.REFERENCE EMPTY>
    <!ATTLIST TABLECELL.REFERENCE
        %CIMName;
        %ReferenceClass;             #REQUIRED
        ISARRAY        (true|false)  "false"
        %ARRAYSIZE;
        CELLPOS         CDATA        #REQUIRED
        SORTPOS          CDATA       #IMPLIED
        SORTDIR         (ASC|DESC)   #IMPLIED>
     """

class TABLEROW_DECLARATION(CIMElement):
    # pylint: disable=invalid-name
    """

    The TABLEROW.DECLARATION element contains a definition for each
    TABLECELL in the TABLEROW.

    <!ELEMENT TABLEROW.DECLARATION (TABLECELL.DECLARATION
                                    | TABLECELL.REFERENCE)*>
    """

class TABLE(CIMElement):
    # pylint: disable=invalid-name
    """
    The TABLE element defines the result of a CIM Query.  A TABLE
    element consists of a TABLEROW.DECLARATION followed by 0 or more
    rows.

    <!ELEMENT TABLE(TABLEROW.DECLARATION,(TABLEROW)*)>
    """

class TABLEROW(CIMElement):
    # pylint: disable=invalid-name
    """

    The TABLEROW element defines the values for a single row of a
    table.  A value for each cell of the row MUST be specified.  If a
    value has no assigned value, the VALUE.NULL element MUST be used.

    <!ELEMENT TABLEROW (VALUE | VALUE.ARRAY | VALUE.REFERENCE |
                        VALUE.REFARRAY | VALUE.NULL)*>
    """

# Message elements

class MESSAGE(CIMElement):
    # pylint: disable=invalid-name
    """
    The MESSAGE element models a single CIM message.  This element is
    used as the basis for CIM Operation Messages and CIM Export
    Messages.

    <!ELEMENT MESSAGE (SIMPLEREQ | MULTIREQ | SIMPLERSP | MULTIRSP |
                       SIMPLEEXPREQ | MULTIEXPREQ | SIMPLEEXPRSP |
                       MULTIEXPRSP)>
    <!ATTLIST MESSAGE
	ID CDATA #REQUIRED
	PROTOCOLVERSION CDATA #REQUIRED>
    """

    def __init__(self, data, message_id, protocol_version):
        Element.__init__(self, 'MESSAGE')
        self.setAttribute('ID', message_id)
        self.setAttribute('PROTOCOLVERSION', protocol_version)
        self.appendChild(data)

class MULTIREQ(CIMElement):
    # pylint: disable=invalid-name
    """
    The MULTIREQ element defines a Multiple CIM Operation request.  It
    contains two or more subelements defining the SIMPLEREQ elements
    that make up this multiple request.

    <!ELEMENT MULTIREQ (SIMPLEREQ, SIMPLEREQ+)>
    """

    def __init__(self, data):
        Element.__init__(self, 'MULTIREQ')
        self.appendChildren(data)

class MULTIEXPREQ(CIMElement):
    # pylint: disable=invalid-name
    """
    The MULTIEXPREQ element defines a Multiple CIM Export request.  It
    contains two or more subelements defining the SIMPLEEXPREQ
    elements that make up this multiple request.

    <!ELEMENT MULTIEXPREQ (SIMPLEEXPREQ, SIMPLEEXPREQ+)>
    """

    def __init__(self, data):
        Element.__init__(self, 'MULTIEXPREQ')
        self.appendChildren(data)

class SIMPLEREQ(CIMElement):
    # pylint: disable=invalid-name
    """
    The SIMPLEREQ element defines a Simple CIM Operation request.  It
    contains either a METHODCALL (extrinsic method) element or an
    IMETHODCALL (intrinsic method) element.

    <!ELEMENT SIMPLEREQ (IMETHODCALL | METHODCALL)>
    """

    def __init__(self, data):
        Element.__init__(self, 'SIMPLEREQ')
        self.appendChild(data)

class SIMPLEEXPREQ(CIMElement):
    # pylint: disable=invalid-name
    """
    The SIMPLEEXPREQ element defines a Simple CIM Export request.  It
    contains an EXPMETHODCALL (export method).

    <!ELEMENT SIMPLEEXPREQ (EXPMETHODCALL)>
    """

    def __init__(self, data):
        Element.__init__(self, 'SIMPLEEXPREQ')
        self.appendChild(data)

class IMETHODCALL(CIMElement):
    # pylint: disable=invalid-name
    """
    The IMETHODCALL element defines a single intrinsic method
    invocation.  It specifies the target local namespace, followed by
    zero or more IPARAMVALUE subelements as the parameter values to be
    passed to the method. If the RESPONSEDESTINATION element is
    specified, the intrinsic method call MUST be interpreted as an
    asynchronous method call.

    <!ELEMENT IMETHODCALL (LOCALNAMESPACEPATH, IPARAMVALUE*,
                           RESPONSEDESTINATION?)>
    <!ATTLIST IMETHODCALL
	%CIMName;>
    """

    def __init__(self, name, localnamespacepath, iparamvalues=[],
                 responsedestination=None):
        Element.__init__(self, 'IMETHODCALL')
        self.setName(name)
        self.appendChild(localnamespacepath)
        self.appendChildren(iparamvalues)
        self.appendOptionalChild(responsedestination)

class METHODCALL(CIMElement):
    # pylint: disable=invalid-name
    """
    The METHODCALL element defines a single method invocation on a
    Class or Instance.  It specifies the local path of the target
    Class or Instance, followed by zero or more PARAMVALUE subelements
    as the parameter values to be passed to the method. If the
    RESPONSEDESTINATION element is specified, the method call MUST be
    interpreted as an asynchronous method call.

    <!ELEMENT METHODCALL ((LOCALINSTANCEPATH | LOCALCLASSPATH), PARAMVALUE*,
                          RESPONSEDESTINATION?)>
    <!ATTLIST METHODCALL
	%CIMName;>
    """

    def __init__(self, name, localpath, paramvalues=[],
                 responsedestination=None):
        Element.__init__(self, 'METHODCALL')
        self.setName(name)
        self.appendChild(localpath)
        self.appendChildren(paramvalues)
        self.appendOptionalChild(responsedestination)

class EXPMETHODCALL(CIMElement):
    # pylint: disable=invalid-name
    """
    The EXPMETHODCALL element defines a single export method
    invocation.  It specifies zero or more  <EXPPARAMVALUE>
    subelements as the parameter values to be passed to the method.

    <!ELEMENT EXPMETHODCALL (EXPPARAMVALUE*)>
    <!ATTLIST EXPMETHODCALL
	%CIMName;>
    """

    def __init__(self, name, params=[]):
        Element.__init__(self, 'EXPMETHODCALL')
        self.setName(name)
        self.appendChildren(params)

class PARAMVALUE(CIMElement):
    # pylint: disable=invalid-name
    """
    The PARAMVALUE element defines a single extrinsic method named
    parameter value. If no subelement is present this indicates that
    no value has been supplied for this parameter.

    <!ELEMENT PARAMVALUE (VALUE | VALUE.REFERENCE | VALUE.ARRAY |
                          VALUE.REFARRAY)?>
    <!ATTLIST PARAMVALUE
	%CIMName;
        %ParamType;    #IMPLIED>
    """

    def __init__(self, name, data=None, paramtype=None,
                 embedded_object=None):
        Element.__init__(self, 'PARAMVALUE')
        self.setName(name)
        self.setOptionalAttribute('PARAMTYPE', paramtype)
        self.setOptionalAttribute('EmbeddedObject', embedded_object)
        # See the note on EmbeddedObject in PROPERTY().
        self.appendOptionalChild(data)

class IPARAMVALUE(CIMElement):
    # pylint: disable=invalid-name
    """
    The IPARAMVALUE element defines a single intrinsic method named
    parameter value. If no subelement is present this indicates that
    no value has been supplied for this parameter.

    <!ELEMENT IPARAMVALUE (VALUE | VALUE.ARRAY | VALUE.REFERENCE |
                           INSTANCENAME | CLASSNAME | QUALIFIER.DECLARATION |
                           CLASS | INSTANCE | VALUE.NAMEDINSTANCE)?>
    <!ATTLIST IPARAMVALUE
	%CIMName;>
    """

    def __init__(self, name, data=None):
        Element.__init__(self, 'IPARAMVALUE')
        self.setName(name)
        self.appendOptionalChild(data)

class EXPPARAMVALUE(CIMElement):
    # pylint: disable=invalid-name
    """
    The EXPPARAMVALUE element defines a single export method named
    parameter value.  If no subelement is present this indicates that
    no value has been supplied for this parameter.

    <!ELEMENT EXPPARAMVALUE (INSTANCE? | VALUE? | METHODRESPONSE? |
                             IMETHODRESPONSE?)>
    <!ATTLIST EXPPARAMVALUE
	%CIMName;
        %ParamType;  #IMPLIED>
    """

    def __init__(self, name, data=None, param_type=None):
        Element.__init__(self, 'EXPPARAMVALUE')
        self.setName(name)
        self.setOptionalAttribute('PARAMTYPE', param_type)
        self.appendOptionalChild(data)

class MULTIRSP(CIMElement):
    # pylint: disable=invalid-name
    """
    The MULTIRSP element defines a Multiple CIM Operation response.
    It contains two or more subelements defining the SIMPLERSP
    elements that make up this multiple response.

    <!ELEMENT MULTIRSP (SIMPLERSP, SIMPLERSP+)>
    """

    def __init__(self, data):
        Element.__init__(self, 'MULTIRSP')
        self.appendChildren(data)

class MULTIEXPRSP(CIMElement):
    # pylint: disable=invalid-name
    """
    The MULTIEXPRSP element defines a Multiple CIM Export response.
    It contains two or more subelements defining the SIMPLEEXPRSP
    elements that make up this multiple response.

    <!ELEMENT MULTIEXPRSP (SIMPLEEXPRSP, SIMPLEEXPRSP+)>
    """

    def __init__(self, data):
        Element.__init__(self, 'MULTIEXPRSP')
        self.appendChildren(data)

class SIMPLERSP(CIMElement):
    # pylint: disable=invalid-name
    """
    The SIMPLERSP element defines a Simple CIM Operation response.  It
    contains either a METHODRESPONSE (for extrinsic methods),
    IMETHODRESPONSE (for intrinsic methods) or a SIMPLEREQACK
    subelement.

    <!ELEMENT SIMPLERSP (METHODRESPONSE | IMETHODRESPONSE | SIMPLEREQACK)>
    """

    def __init__(self, data):
        Element.__init__(self, 'SIMPLERSP')
        self.appendChild(data)

class SIMPLEEXPRSP(CIMElement):
    # pylint: disable=invalid-name
    """
    The SIMPLEEXPRSP element defines a Simple CIM Export response.  It
    contains either a EXPMETHODRESPONSE (for export methods)
    subelement.

    <!ELEMENT SIMPLEEXPRSP (EXPMETHODRESPONSE)>
    """

    def __init__(self, data):
        Element.__init__(self, 'SIMPLEEXPRSP')
        self.appendChild(data)

class METHODRESPONSE(CIMElement):
    # pylint: disable=invalid-name
    """
    The METHODRESPONSE defines the response to a single CIM extrinsic
    method invocation.  It contains either an ERROR subelement (to
    report a fundamental error which prevented the method from
    executing), or a combination of an optional return value and zero
    or more out parameter values.

    <!ELEMENT METHODRESPONSE (ERROR | (RETURNVALUE?, PARAMVALUE*))>
    <!ATTLIST METHODRESPONSE
	%CIMName;>
    """

    def __init__(self, name, data=None):
        Element.__init__(self, 'METHODRESPONSE')
        self.setName(name)
        if data is not None:
            if isinstance(data, (tuple, list)):
                self.appendChildren(data)
            else:
                self.appendChild(data)

class EXPMETHODRESPONSE(CIMElement):
    # pylint: disable=invalid-name
    """
    The EXPMETHODRESPONSE defines the response to a single export
    method invocation.  It contains either an ERROR subelement (to
    report a fundamental error which prevented the method from
    executing), or an optional return value.

    <!ELEMENT EXPMETHODRESPONSE (ERROR | IRETURNVALUE?)>
    <!ATTLIST EXPMETHODRESPONSE
	%CIMName;>
    """

    def __init__(self, name, data=None):
        Element.__init__(self, 'EXPMETHODRESPONSE')
        self.setName(name)
        self.appendOptionalChild(data)

class IMETHODRESPONSE(CIMElement):
    # pylint: disable=invalid-name
    """
    The IMETHODRESPONSE defines the response to a single intrinsic CIM
    method invocation.  It contains either an ERROR subelement (to
    report a fundamental error which prevented the method from
    executing), or an optional return value.

    <!ELEMENT IMETHODRESPONSE (ERROR | IRETURNVALUE?)>
    <!ATTLIST IMETHODRESPONSE
	%CIMName;>
    """

    def __init__(self, name, data=None):
        Element.__init__(self, 'IMETHODRESPONSE')
        self.setName(name)
        self.appendOptionalChild(data)

class ERROR(CIMElement):
    # pylint: disable=invalid-name
    """
    The ERROR element is used to define a fundamental error which
    prevented a method from executing normally. It consists of a
    status code, an optional description and zero or more instances
    containing detailed information about the error.

    <!ELEMENT ERROR (INSTANCE*)>
    <!ATTLIST ERROR
	CODE CDATA #REQUIRED
	DESCRIPTION CDATA #IMPLIED>
    """

    def __init__(self, code, description=None, instances=[]):
        Element.__init__(self, 'ERROR')
        self.setAttribute('CODE', code)
        self.setOptionalAttribute('DESCRIPTION', description)
        self.appendChildren(instances)

class RETURNVALUE(CIMElement):
    # pylint: disable=invalid-name
    """
    The RETURNVALUE element specifies the value returned from an
    extrinsic method call.

    <!ELEMENT RETURNVALUE (VALUE | VALUE.REFERENCE)>
    <!ATTLIST RETURNVALUE
        %ParamType;     #IMPLIED>
    """

    def __init__(self, data, param_type=None):
        Element.__init__(self, 'RETURNVALUE')
        self.setOptionalAttribute('PARAMTYPE', param_type)
        self.appendChild(data)

class IRETURNVALUE(CIMElement):
    # pylint: disable=invalid-name
    """
    The IRETURNVALUE element specifies the value returned from an
    intrinsic method call.

    <!ELEMENT IRETURNVALUE (CLASSNAME* | INSTANCENAME* | VALUE* |
                            VALUE.OBJECTWITHPATH* |
                            VALUE.OBJECTWITHLOCALPATH* | VALUE.OBJECT* |
                            OBJECTPATH* | QUALIFIER.DECLARATION* |
                            VALUE.ARRAY? | VALUE.REFERENCE? | CLASS* |
                            INSTANCE* | VALUE.NAMEDINSTANCE*)>
    """

    def __init__(self, data):
        Element.__init__(self, 'IRETURNVALUE')
        self.appendOptionalChild(data)

class RESPONSEDESTINATION(CIMElement):
    # pylint: disable=invalid-name
    """
    The RESPONSEDESTINATION element contains an instance that
    describes the desired destination for the response.

    <!ELEMENT RESPONSEDESTINATON (INSTANCE)>
    """

    def __init__(self, data):
        Element.__init__(self, 'RESPONSEDESTINATON')
        self.appendChild(data)

class SIMPLEREQACK(CIMElement):
    # pylint: disable=invalid-name
    """
    The SIMPLEREQACK defines the acknowledgement response to a Simple
    CIM Operation asynchronous request. The ERROR subelement is used
    to report a fundamental error which prevented the asynchronous
    request from being initiated.

    <!ELEMENT SIMPLEREQACK (ERROR?)>
    <!ATTLIST SIMPLEREQACK
        INSTANCEID CDATA     #REQUIRED>
    """

    def __init__(self, instanceid, data):
        Element.__init__(self, 'SIMPLEREQACK')
        self.setOptionalAttribute('INSTANCEID', instanceid)
        self.appendOptionalChild(data)
