# This file contains the ReadFileChunk class implementation, by which we can stream the contents of the file in the
# parts/chunks. This class will allow you to specify the size and stream the data without using a chunked
# transfer-encoding. This implementation has been taken from https://github.com/boto/s3transfer/blob/develop/s3transfer/__init__.py

import os


class ReadFileChunk(object):
    def __init__(self, fileobj, start_byte, chunk_size, full_file_size,
                 callback=None, enable_callback=True):
        """
        Given a file object shown below:
            |___________________________________________________|
            0          |                 |                 full_file_size
                       |----chunk_size---|
                 start_byte
        :type fileobj: file
        :param fileobj: File like object
        :type start_byte: int
        :param start_byte: The first byte from which to start reading.
        :type chunk_size: int
        :param chunk_size: The max chunk size to read.  Trying to read
            pass the end of the chunk size will behave like you've
            reached the end of the file.
        :type full_file_size: int
        :param full_file_size: The entire content length associated
            with ``fileobj``.
        :type callback: function(amount_read)
        :param callback: Called whenever data is read from this object.
        """
        self._fileobj = fileobj
        self._start_byte = start_byte
        self._size = self._calculate_file_size(
            self._fileobj, requested_size=chunk_size,
            start_byte=start_byte, actual_file_size=full_file_size)
        self._fileobj.seek(self._start_byte)
        self._amount_read = 0
        self._callback = callback
        self._callback_enabled = enable_callback

    @classmethod
    def from_filename(cls, filename, start_byte, chunk_size, callback=None,
                      enable_callback=True):
        """Convenience factory function to create from a filename.
        :type start_byte: int
        :param start_byte: The first byte from which to start reading.
        :type chunk_size: int
        :param chunk_size: The max chunk size to read.  Trying to read
            pass the end of the chunk size will behave like you've
            reached the end of the file.
        :type full_file_size: int
        :param full_file_size: The entire content length associated
            with ``fileobj``.
        :type callback: function(amount_read)
        :param callback: Called whenever data is read from this object.
        :type enable_callback: bool
        :param enable_callback: Indicate whether to invoke callback
            during read() calls.
        :rtype: ``ReadFileChunk``
        :return: A new instance of ``ReadFileChunk``
        """
        f = open(filename, 'rb')
        file_size = os.fstat(f.fileno()).st_size
        return cls(f, start_byte, chunk_size, file_size, callback,
                   enable_callback)

    def _calculate_file_size(self, fileobj, requested_size, start_byte,
                             actual_file_size):
        max_chunk_size = actual_file_size - start_byte
        return min(max_chunk_size, requested_size)

    def read(self, amount=None):
        if amount is None:
            amount_to_read = self._size - self._amount_read
        else:
            amount_to_read = min(self._size - self._amount_read, amount)
        data = self._fileobj.read(amount_to_read)
        self._amount_read += len(data)
        if self._callback is not None and self._callback_enabled:
            self._callback(len(data))
        return data

    def enable_callback(self):
        self._callback_enabled = True

    def disable_callback(self):
        self._callback_enabled = False

    def seek(self, where):
        self._fileobj.seek(self._start_byte + where)
        if self._callback is not None and self._callback_enabled:
            # To also rewind the callback() for an accurate progress report
            self._callback(where - self._amount_read)
        self._amount_read = where

    def close(self):
        self._fileobj.close()

    def tell(self):
        return self._amount_read

    def __len__(self):
        # __len__ is defined because requests will try to determine the length
        # of the stream to set a content length.  In the normal case
        # of the file it will just stat the file, but we need to change that
        # behavior.  By providing a __len__, requests will use that instead
        # of stat'ing the file.
        return self._size

    def __enter__(self):
        return self

    def __exit__(self, *args, **kwargs):
        self.close()

    def __iter__(self):
        # This is a workaround for http://bugs.python.org/issue17575
        # Basically httplib will try to iterate over the contents, even
        # if its a file like object.  This wasn't noticed because we've
        # already exhausted the stream so iterating over the file immediately
        # stops, which is what we're simulating here.
        return iter([])


def open_file_chunk_reader(filename, start_byte, size, callback):
    return ReadFileChunk.from_filename(filename, start_byte, size, callback)
