﻿// Copyright (c) Microsoft.  All Rights Reserved.  Licensed under the Apache License, Version 2.0.  See License.txt in the project root for license information.

using System.Collections.Generic;
using System.Threading;
using Microsoft.CodeAnalysis.CSharp.Extensions.ContextQuery;
using Microsoft.CodeAnalysis.CSharp.Extensions;
using Microsoft.CodeAnalysis.CSharp.Utilities;

namespace Microsoft.CodeAnalysis.CSharp.Completion.KeywordRecommenders
{
    internal class ReadOnlyKeywordRecommender : AbstractSyntacticSingleKeywordRecommender
    {
        private static readonly ISet<SyntaxKind> s_validMemberModifiers = new HashSet<SyntaxKind>(SyntaxFacts.EqualityComparer)
            {
                SyntaxKind.NewKeyword,
                SyntaxKind.PublicKeyword,
                SyntaxKind.ProtectedKeyword,
                SyntaxKind.InternalKeyword,
                SyntaxKind.PrivateKeyword,
                SyntaxKind.StaticKeyword,
            };

        public ReadOnlyKeywordRecommender()
            : base(SyntaxKind.ReadOnlyKeyword)
        {
        }

        protected override bool IsValidContext(int position, CSharpSyntaxContext context, CancellationToken cancellationToken)
        {
            return
                context.IsGlobalStatementContext ||
                IsRefReadOnlyContext(position, context) ||
                context.SyntaxTree.IsGlobalMemberDeclarationContext(context.Position, SyntaxKindSet.AllGlobalMemberModifiers, cancellationToken) ||
                context.IsMemberDeclarationContext(
                    validModifiers: s_validMemberModifiers,
                    validTypeDeclarations: SyntaxKindSet.ClassStructTypeDeclarations,
                    canBePartial: false,
                    cancellationToken: cancellationToken);
        }

        private bool IsRefReadOnlyContext(int position, CSharpSyntaxContext context)
        {
            var previousToken = context.LeftToken.GetPreviousTokenIfTouchingWord(position);

            return previousToken.IsKind(SyntaxKind.RefKeyword) &&
                previousToken.Parent.IsKind(SyntaxKind.RefType) &&
                previousToken.Parent.Parent.IsKind(
                    SyntaxKind.PropertyDeclaration,
                    SyntaxKind.MethodDeclaration,
                    SyntaxKind.DelegateDeclaration,
                    SyntaxKind.IncompleteMember);
        }
    }
}
