//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.
//

//----------------------------------------------------------
// MethodContext.h - Primary structure to store all the EE-JIT details required to replay creation of a method
//                   CompileResult contains the stuff generated by a compilation
//----------------------------------------------------------
#ifndef _MethodContext
#define _MethodContext

#include "runtimedetails.h"
#include "compileresult.h"
#include "lightweightmap.h"
#include "errorhandling.h"

#define METHOD_IDENTITY_INFO_SIZE 0x10000 //We assume that the METHOD_IDENTITY_INFO_SIZE will not exceed 64KB

#define MD5_HASH_BYTE_SIZE 16 //MD5 is 128-bit, so we need 16 bytes to store it
#define MD5_HASH_BUFFER_SIZE 33 //MD5 is 128-bit, so we need 32 chars + 1 char to store null-terminator

class MethodContext
{
public:
#pragma pack(push, 1)
    struct Agnostic_CORINFO_SIG_INFO
    {
        DWORD callConv;
        DWORDLONG retTypeClass;
        DWORDLONG retTypeSigClass;
        DWORD retType;
        DWORD flags;
        DWORD numArgs;
        DWORD sigInst_classInstCount;
        DWORD sigInst_classInst_Index;
        DWORD sigInst_methInstCount;
        DWORD sigInst_methInst_Index;
        DWORDLONG args;
        DWORD pSig;
        DWORD cbSig;
        DWORDLONG scope;
        DWORD token;
    };
    struct Agnostic_CORINFO_METHOD_INFO
    {
        DWORDLONG ftn;
        DWORDLONG scope;
        DWORD ILCode_offset;
        DWORD ILCodeSize;
        DWORD maxStack;
        DWORD EHcount;
        DWORD options;
        DWORD regionKind;
        Agnostic_CORINFO_SIG_INFO args;
        Agnostic_CORINFO_SIG_INFO locals;
    };
    struct Agnostic_CompileMethod
    {
        Agnostic_CORINFO_METHOD_INFO info;
        DWORD flags;
    };
    struct Agnostic_InitClass
    {
        DWORDLONG field;
        DWORDLONG method;
        DWORDLONG context;
        DWORD speculative;
    };
    struct DLDL
    {
        DWORDLONG A;
        DWORDLONG B;
    };
    struct Agnostic_CanInline
    {
        DWORD Restrictions;
        DWORD result;
        DWORD exceptionCode;
    };
    struct Agnostic_GetClassGClayout
    {
        DWORD gcPtrs_Index;
        DWORD len;
        DWORD valCount;
    };
    struct DLD
    {
        DWORDLONG A;
        DWORD B;
    };
    struct Agnostic_CORINFO_RESOLVED_TOKENin
    {
        DWORDLONG tokenContext;
        DWORDLONG tokenScope;
        DWORD token;
        DWORD tokenType;
    };
    struct Agnostic_CORINFO_RESOLVED_TOKENout
    {
        DWORDLONG hClass;
        DWORDLONG hMethod;
        DWORDLONG hField;
        DWORD pTypeSpec_Index;
        DWORD cbTypeSpec;
        DWORD pMethodSpec_Index;
        DWORD cbMethodSpec;
        DWORD exceptionCode;
    };
    struct Agnostic_GetArgType
    {
        Agnostic_CORINFO_SIG_INFO sig;
        DWORDLONG args;
    };
    struct Agnostic_GetArgClass
    {
        Agnostic_CORINFO_SIG_INFO sig;
        DWORDLONG args;
    };
    struct Agnostic_GetBoundaries
    {
        DWORD cILOffsets;
        DWORD pILOffset_offset;
        DWORD implicitBoundaries;
    };
    struct Agnostic_CORINFO_EE_INFO
    {
        struct Agnostic_InlinedCallFrameInfo
        {
            DWORD size;
            DWORD offsetOfGSCookie;
            DWORD offsetOfFrameVptr;
            DWORD offsetOfFrameLink;
            DWORD offsetOfCallSiteSP;
            DWORD offsetOfCalleeSavedFP;
            DWORD offsetOfCallTarget;
            DWORD offsetOfReturnAddress;
        }
        inlinedCallFrameInfo;
        DWORD offsetOfThreadFrame;
        DWORD offsetOfGCState;
        DWORD offsetOfDelegateInstance;
        DWORD offsetOfDelegateFirstTarget;
        DWORD offsetOfSecureDelegateIndirectCell;
        DWORD offsetOfTransparentProxyRP;
        DWORD offsetOfRealProxyServer;
        DWORD offsetOfObjArrayData;
        DWORD sizeOfReversePInvokeFrame;
        DWORD osPageSize;
        DWORD maxUncheckedOffsetForNullObject;
        DWORD targetAbi;
        DWORD osType;
        DWORD osMajor;
        DWORD osMinor;
        DWORD osBuild;
    };
    struct Agnostic_GetFieldAddress
    {
        DWORDLONG ppIndirection;
        DWORDLONG fieldAddress;
        DWORD fieldValue;
    };
    struct Agnostic_CORINFO_RESOLVED_TOKEN
    {
        DWORDLONG tokenContext;
        DWORDLONG tokenScope;
        DWORD token;
        DWORD tokenType;
        DWORDLONG hClass;
        DWORDLONG hMethod;
        DWORDLONG hField;
        DWORD typeSpec_Index;
        DWORD cbTypeSpec;
        DWORD methodSpec_Index;
        DWORD cbMethodSpec;
    };
    struct Agnostic_GetFieldInfo
    {
        Agnostic_CORINFO_RESOLVED_TOKEN ResolvedToken;
        DWORDLONG callerHandle;
        DWORD flags;
    };
    struct Agnostic_CORINFO_HELPER_ARG
    {
        DWORDLONG constant; //one view of a large union of ptr size
        DWORD argType;
    };
    struct Agnostic_CORINFO_HELPER_DESC
    {
        DWORD helperNum;
        DWORD numArgs;
        Agnostic_CORINFO_HELPER_ARG args[CORINFO_ACCESS_ALLOWED_MAX_ARGS];
    };
    struct Agnostic_CORINFO_FIELD_INFO
    {
        DWORD fieldAccessor;
        DWORD fieldFlags;
        DWORD helper;
        DWORD offset;
        DWORD fieldType;
        DWORDLONG structType;
        DWORD accessAllowed;
        Agnostic_CORINFO_HELPER_DESC accessCalloutHelper;
    };
    struct DD
    {
        DWORD A;
        DWORD B;
    };
    struct Agnostic_CanTailCall
    {
        DWORDLONG callerHnd;
        DWORDLONG declaredCalleeHnd;
        DWORDLONG exactCalleeHnd;
        WORD fIsTailPrefix;
    };
    struct Agnostic_Environment
    {
        DWORD name_index;;
        DWORD val_index;
    };
    struct Agnostic_GetCallInfo
    {
        Agnostic_CORINFO_RESOLVED_TOKEN ResolvedToken;
        Agnostic_CORINFO_RESOLVED_TOKEN ConstrainedResolvedToken;
        DWORDLONG callerHandle;
        DWORD flags;
    };
    struct Agnostic_CORINFO_LOOKUP_KIND
    {
        DWORD needsRuntimeLookup;
        DWORD runtimeLookupKind;
        WORD runtimeLookupFlags;
    };
    struct Agnostic_CORINFO_RUNTIME_LOOKUP
    {
        DWORDLONG signature;
        DWORD helper;
        DWORD indirections;
        DWORD testForNull;
        DWORD testForFixup;
        DWORDLONG offsets[CORINFO_MAXINDIRECTIONS];
    };
    struct Agnostic_CORINFO_CONST_LOOKUP
    {
        DWORD accessType;
        DWORDLONG  handle; // actually a union of two pointer sized things
    };
    struct Agnostic_CORINFO_LOOKUP
    {
        Agnostic_CORINFO_LOOKUP_KIND     lookupKind;
        Agnostic_CORINFO_RUNTIME_LOOKUP  runtimeLookup; //This and constLookup actually a union, but with different layouts.. :-| copy the right one based on lookupKinds value
        Agnostic_CORINFO_CONST_LOOKUP    constLookup;
    };
    struct Agnostic_CORINFO_CALL_INFO
    {
        DWORDLONG hMethod;
        DWORD methodFlags;
        DWORD classFlags;
        Agnostic_CORINFO_SIG_INFO sig;
        DWORD verMethodFlags;
        Agnostic_CORINFO_SIG_INFO verSig;
        DWORD accessAllowed;
        Agnostic_CORINFO_HELPER_DESC callsiteCalloutHelper;
        DWORD thisTransform;
        DWORD kind;
        DWORD nullInstanceCheck;
        DWORDLONG  contextHandle;
        DWORD exactContextNeedsRuntimeLookup;
        Agnostic_CORINFO_LOOKUP      stubLookup;//first view of union.  others are matching or subordinate
        Agnostic_CORINFO_CONST_LOOKUP instParamLookup;
        DWORD secureDelegateInvoke;
        DWORD exceptionCode;
    };
    struct Agnostic_GetMethodInfo
    {
        Agnostic_CORINFO_METHOD_INFO info;
        bool result;
        DWORD exceptionCode;
    };
    struct Agnostic_FindSig
    {
        DWORDLONG module;
        DWORD sigTOK;
        DWORDLONG context;
    };
    struct Agnostic_PInvokeMarshalingRequired
    {
        DWORDLONG method;
        Agnostic_CORINFO_SIG_INFO callSiteSig;
    };
    struct Agnostic_CORINFO_EH_CLAUSE
    {
        DWORD Flags;
        DWORD TryOffset;
        DWORD TryLength;
        DWORD HandlerOffset;
        DWORD HandlerLength;
        DWORD ClassToken;//first view of a two dword union
    };
    struct Agnostic_GetVars
    {
        DWORD cVars;
        DWORD vars_offset;
        DWORD extendOthers;
    };
    struct Agnostic_CanAccessClassIn
    {
        Agnostic_CORINFO_RESOLVED_TOKEN ResolvedToken;
        DWORDLONG callerHandle;
    };
    struct Agnostic_CanAccessClassOut
    {
        Agnostic_CORINFO_HELPER_DESC AccessHelper;
        DWORD result;
    };
    struct Agnostic_AppendClassName
    {
        DWORDLONG classHandle;
        DWORD     fNamespace;
        DWORD     fFullInst;
        DWORD     fAssembly;
    };
    struct Agnostic_CheckMethodModifier
    {
        DWORDLONG hMethod;
        DWORD modifier;
        DWORD fOptional;
    };
    struct Agnostic_EmbedGenericHandle
    {
        Agnostic_CORINFO_RESOLVED_TOKEN ResolvedToken;
        DWORD fEmbedParent;
    };
    struct Agnostic_CORINFO_GENERICHANDLE_RESULT
    {

        Agnostic_CORINFO_LOOKUP lookup;
        DWORDLONG compileTimeHandle;
        DWORD handleType;
    };
    struct Agnostic_GetDelegateCtorIn
    {
        DWORDLONG methHnd;
        DWORDLONG clsHnd;
        DWORDLONG targetMethodHnd;
    };
    struct Agnostic_DelegateCtorArgs
    {
        DWORDLONG pMethod;
        DWORDLONG pArg3;
        DWORDLONG pArg4;
        DWORDLONG pArg5;
    };
    struct Agnostic_GetDelegateCtorOut
    {
        Agnostic_DelegateCtorArgs CtorData;
        DWORDLONG result;
    };
    struct Agnostic_FindCallSiteSig
    {
        DWORDLONG module;
        DWORD methTok;
        DWORDLONG context;
    };
    struct Agnostic_GetNewHelper
    {
        Agnostic_CORINFO_RESOLVED_TOKEN ResolvedToken;
        DWORDLONG callerHandle;
    };
    struct Agnostic_GetCastingHelper
    {
        Agnostic_CORINFO_RESOLVED_TOKEN ResolvedToken;
        DWORD fThrowing;
    };
    struct Agnostic_GetClassModuleIdForStatics
    {
        DWORDLONG Module;
        DWORDLONG pIndirection;
        DWORDLONG result;
    };
    struct Agnostic_IsCompatibleDelegate
    {
        DWORDLONG objCls;
        DWORDLONG methodParentCls;
        DWORDLONG method;
        DWORDLONG delegateCls;
    };
    struct Agnostic_GetBBProfileData
    {
        DWORD count;
        DWORD profileBuffer_index;
        DWORD numRuns;
        DWORD result;
    };
    struct Agnostic_GetProfilingHandle
    {
        DWORD bHookFunction;
        DWORDLONG ProfilerHandle;
        DWORD bIndirectedHandles;
    };
    struct Agnostic_GetTailCallCopyArgsThunk
    {
        Agnostic_CORINFO_SIG_INFO Sig;
        DWORD flags;
    };
    struct Agnostic_GetArgClass_Value
    {
        DWORDLONG result;
        DWORD exceptionCode;
    };
    struct Agnostic_GetArgType_Value
    {
        DWORDLONG vcTypeRet;
        DWORD result;
        DWORD exceptionCode;
    };

    // Agnostic_ConfigIntInfo combines as a single key the name
    // and defaultValue of a integer config query.
    // Note: nameIndex is treated as a DWORD index to the name string.
    struct Agnostic_ConfigIntInfo
    {
        DWORD nameIndex;
        DWORD defaultValue;
    };

    // SYSTEMV_AMD64_CORINFO_STRUCT_REG_PASSING_DESCRIPTOR
    struct Agnostic_GetSystemVAmd64PassStructInRegisterDescriptor
    {
        DWORD   passedInRegisters; // Whether the struct is passable/passed (this includes struct returning) in registers.
        DWORD   eightByteCount;    // Number of eightbytes for this struct.
        DWORD   eightByteClassifications[CLR_SYSTEMV_MAX_EIGHTBYTES_COUNT_TO_PASS_IN_REGISTERS]; // The eightbytes type classification.
        DWORD   eightByteSizes[CLR_SYSTEMV_MAX_EIGHTBYTES_COUNT_TO_PASS_IN_REGISTERS];           // The size of the eightbytes (an eightbyte could include padding. This represents the no padding size of the eightbyte).
        DWORD   eightByteOffsets[CLR_SYSTEMV_MAX_EIGHTBYTES_COUNT_TO_PASS_IN_REGISTERS];         // The start offset of the eightbytes (in bytes).
        DWORD   result;
    };

#pragma pack(pop)

    MethodContext();

private:

    MethodContext(HANDLE hFile);
    MethodContext(unsigned char *buff, unsigned int totalLen);

    void MethodInitHelper(unsigned char *buff, unsigned int totalLen);
    void MethodInitHelperFile(HANDLE hFile);

    bool Initialize(int loadedCount, unsigned char* buff, DWORD size);
    bool Initialize(int loadedCount, HANDLE hFile);

    int dumpMD5HashToBuffer(BYTE *pBuffer, int bufLen, char *buff, int len);

public:

    static bool Initialize(int loadedCount, unsigned char* buff, DWORD size, /* OUT */ MethodContext** ppmc);
    static bool Initialize(int loadedCount, HANDLE hFile, /* OUT */ MethodContext** ppmc);
    ~MethodContext();
    void Destroy();

    bool Equal(MethodContext *other);
    unsigned int saveToFile(HANDLE hFile);
    unsigned int calculateFileSize();
    unsigned int calculateRawFileSize();
    void dumpToConsole(int mcNumber = -1); // if mcNumber is not -1, display the method context number before the dumped info
    int dumpStatToBuffer(char *buff, int len);
    static int dumpStatTitleToBuffer(char *buff, int len);
    int methodSize;

    int dumpMethodIdentityInfoToBuffer(char *buff, int len);
    int dumpMethodMD5HashToBuffer(char *buff, int len);

    void recGlobalContext(const MethodContext& other);

    void recEnvironment();
    void dmpEnvironment(DWORD key, const Agnostic_Environment& value);
    void repEnvironmentSet();
    void repEnvironmentUnset();
    int repEnvironmentGetCount();
    int repGetTestID();

    void recCompileMethod(CORINFO_METHOD_INFO *info, unsigned flags);
    void dmpCompileMethod(DWORD key, const Agnostic_CompileMethod& value);
    void repCompileMethod(CORINFO_METHOD_INFO *info, unsigned *flags);

    void recGetMethodClass(CORINFO_METHOD_HANDLE methodHandle, CORINFO_CLASS_HANDLE classHandle);
    void dmpGetMethodClass(DWORDLONG key, DWORDLONG value);
    CORINFO_CLASS_HANDLE repGetMethodClass(CORINFO_METHOD_HANDLE methodHandle);

    void recGetClassAttribs(CORINFO_CLASS_HANDLE classHandle, DWORD attribs);
    void dmpGetClassAttribs(DWORDLONG key, DWORD value);
    DWORD repGetClassAttribs(CORINFO_CLASS_HANDLE classHandle);

    void recGetMethodAttribs(CORINFO_METHOD_HANDLE methodHandle, DWORD attribs);
    void dmpGetMethodAttribs(DWORDLONG key, DWORD value);
    DWORD repGetMethodAttribs(CORINFO_METHOD_HANDLE methodHandle);

    void recGetVars(CORINFO_METHOD_HANDLE ftn, ULONG32 *cVars, ICorDebugInfo::ILVarInfo **vars, bool *extendOthers);
    void dmpGetVars(DWORDLONG key, const Agnostic_GetVars& value);
    void repGetVars(CORINFO_METHOD_HANDLE ftn, ULONG32 *cVars, ICorDebugInfo::ILVarInfo **vars, bool *extendOthers);

    void recGetBoundaries(CORINFO_METHOD_HANDLE ftn, unsigned int *cILOffsets, DWORD **pILOffsets, ICorDebugInfo::BoundaryTypes *implictBoundaries);
    void dmpGetBoundaries(DWORDLONG key, const Agnostic_GetBoundaries& value);
    void repGetBoundaries(CORINFO_METHOD_HANDLE ftn, unsigned int *cILOffsets, DWORD **pILOffsets, ICorDebugInfo::BoundaryTypes *implictBoundaries);

    void recInitClass(CORINFO_FIELD_HANDLE field, CORINFO_METHOD_HANDLE method, CORINFO_CONTEXT_HANDLE context, BOOL speculative, CorInfoInitClassResult result);
    void dmpInitClass(const Agnostic_InitClass& key, DWORD value);
    CorInfoInitClassResult repInitClass(CORINFO_FIELD_HANDLE field, CORINFO_METHOD_HANDLE method, CORINFO_CONTEXT_HANDLE context, BOOL speculative);

    void recGetMethodName(CORINFO_METHOD_HANDLE ftn, char *methodname, const char **moduleName);
    void dmpGetMethodName(DLD key, DD value);
    const char *repGetMethodName(CORINFO_METHOD_HANDLE ftn, const char **moduleName);

    void MethodContext::recGetJitFlags(CORJIT_FLAGS *jitFlags, DWORD sizeInBytes, DWORD result);
    void MethodContext::dmpGetJitFlags(DWORD key, DD value);
    DWORD MethodContext::repGetJitFlags(CORJIT_FLAGS *jitFlags, DWORD sizeInBytes);

    void recGetJitTimeLogFilename(LPCWSTR tempFileName);
    void dmpGetJitTimeLogFilename(DWORD key, DWORD value);
    LPCWSTR repGetJitTimeLogFilename();

    void recCanInline(CORINFO_METHOD_HANDLE callerHnd, CORINFO_METHOD_HANDLE calleeHnd, DWORD *pRestrictions, CorInfoInline response, DWORD exceptionCode);
    void dmpCanInline(DLDL key, const Agnostic_CanInline& value);
    CorInfoInline repCanInline(CORINFO_METHOD_HANDLE callerHnd, CORINFO_METHOD_HANDLE calleeHnd, DWORD* pRestrictions, DWORD *exceptionCode);

    void recResolveToken(CORINFO_RESOLVED_TOKEN * pResolvedToken, DWORD exceptionCode);
    void dmpResolveToken(const Agnostic_CORINFO_RESOLVED_TOKENin& key, const Agnostic_CORINFO_RESOLVED_TOKENout& value);
    void repResolveToken(CORINFO_RESOLVED_TOKEN * pResolvedToken, DWORD *exceptionCode);

    void recTryResolveToken(CORINFO_RESOLVED_TOKEN * pResolvedToken, bool success);
    void dmpTryResolveToken(const Agnostic_CORINFO_RESOLVED_TOKENin& key, const Agnostic_CORINFO_RESOLVED_TOKENout& value);
    bool repTryResolveToken(CORINFO_RESOLVED_TOKEN * pResolvedToken);

    void recGetCallInfo(CORINFO_RESOLVED_TOKEN *pResolvedToken, CORINFO_RESOLVED_TOKEN *pConstrainedResolvedToken,
        CORINFO_METHOD_HANDLE callerHandle, CORINFO_CALLINFO_FLAGS flags, CORINFO_CALL_INFO *pResult, DWORD exceptionCode);
    void dmpGetCallInfo(const Agnostic_GetCallInfo& key, const Agnostic_CORINFO_CALL_INFO& value);
    void repGetCallInfo(CORINFO_RESOLVED_TOKEN *pResolvedToken, CORINFO_RESOLVED_TOKEN *pConstrainedResolvedToken,
        CORINFO_METHOD_HANDLE callerHandle, CORINFO_CALLINFO_FLAGS flags, CORINFO_CALL_INFO *pResult, DWORD *exceptionCode);
    void repGetCallInfoFromMethodHandle(CORINFO_METHOD_HANDLE methodHandle, CORINFO_CALL_INFO *pResult);

    void recGetIntrinsicID(CORINFO_METHOD_HANDLE method, bool* pMustExpand, CorInfoIntrinsics result);
    void dmpGetIntrinsicID(DWORDLONG key, DD value);
    CorInfoIntrinsics repGetIntrinsicID(CORINFO_METHOD_HANDLE method, bool* pMustExpand);

    void recGetUnmanagedCallConv(CORINFO_METHOD_HANDLE method, CorInfoUnmanagedCallConv result);
    void dmpGetUnmanagedCallConv(DWORDLONG key, DWORD result);
    CorInfoUnmanagedCallConv repGetUnmanagedCallConv(CORINFO_METHOD_HANDLE method);

    void recIsInstantiationOfVerifiedGeneric(CORINFO_METHOD_HANDLE method, CorInfoInstantiationVerification result);
    void dmpIsInstantiationOfVerifiedGeneric(DWORDLONG key, DWORD value);
    CorInfoInstantiationVerification repIsInstantiationOfVerifiedGeneric(CORINFO_METHOD_HANDLE method);

    void recAsCorInfoType(CORINFO_CLASS_HANDLE cls, CorInfoType result);
    void dmpAsCorInfoType(DWORDLONG key, DWORD value);
    CorInfoType repAsCorInfoType(CORINFO_CLASS_HANDLE cls);

    void recIsValueClass(CORINFO_CLASS_HANDLE cls, BOOL result);
    void dmpIsValueClass(DWORDLONG key, DWORD value);
    BOOL repIsValueClass(CORINFO_CLASS_HANDLE cls);

    void recIsStructRequiringStackAllocRetBuf(CORINFO_CLASS_HANDLE cls, BOOL result);
    void dmpIsStructRequiringStackAllocRetBuf(DWORDLONG key, DWORD value);
    BOOL repIsStructRequiringStackAllocRetBuf(CORINFO_CLASS_HANDLE cls);

    void recGetClassSize(CORINFO_CLASS_HANDLE cls, unsigned result);
    void dmpGetClassSize(DWORDLONG key, DWORD val);
    unsigned repGetClassSize(CORINFO_CLASS_HANDLE cls);

    void recGetClassNumInstanceFields(CORINFO_CLASS_HANDLE cls, unsigned result);
    void dmpGetClassNumInstanceFields(DWORDLONG key, DWORD value);
    unsigned repGetClassNumInstanceFields(CORINFO_CLASS_HANDLE cls);

    void recGetNewArrHelper(CORINFO_CLASS_HANDLE arrayCls, CorInfoHelpFunc result);
    void dmpGetNewArrHelper(DWORDLONG key, DWORD value);
    CorInfoHelpFunc repGetNewArrHelper(CORINFO_CLASS_HANDLE arrayCls);

    void recGetSharedCCtorHelper(CORINFO_CLASS_HANDLE clsHnd, CorInfoHelpFunc result);
    void dmpGetSharedCCtorHelper(DWORDLONG key, DWORD value);
    CorInfoHelpFunc repGetSharedCCtorHelper(CORINFO_CLASS_HANDLE clsHnd);

    void recGetSecurityPrologHelper(CORINFO_METHOD_HANDLE ftn, CorInfoHelpFunc result);
    void dmpGetSecurityPrologHelper(DWORDLONG key, DWORD value);
    CorInfoHelpFunc repGetSecurityPrologHelper(CORINFO_METHOD_HANDLE ftn);

    void recGetTypeForBox(CORINFO_CLASS_HANDLE cls, CORINFO_CLASS_HANDLE result);
    void dmpGetTypeForBox(DWORDLONG key, DWORDLONG value);
    CORINFO_CLASS_HANDLE  repGetTypeForBox(CORINFO_CLASS_HANDLE cls);

    void recGetBoxHelper(CORINFO_CLASS_HANDLE cls, CorInfoHelpFunc result);
    void dmpGetBoxHelper(DWORDLONG key, DWORD value);
    CorInfoHelpFunc repGetBoxHelper(CORINFO_CLASS_HANDLE cls);

    void recGetBuiltinClass(CorInfoClassId classId, CORINFO_CLASS_HANDLE result);
    void dmpGetBuiltinClass(DWORD key, DWORDLONG value);
    CORINFO_CLASS_HANDLE repGetBuiltinClass(CorInfoClassId classId);

    void recGetTypeForPrimitiveValueClass(CORINFO_CLASS_HANDLE cls, CorInfoType result);
    void dmpGetTypeForPrimitiveValueClass(DWORDLONG key, DWORD value);
    CorInfoType repGetTypeForPrimitiveValueClass(CORINFO_CLASS_HANDLE cls);

    void recGetParentType(CORINFO_CLASS_HANDLE cls, CORINFO_CLASS_HANDLE result);
    void dmpGetParentType(DWORDLONG key, DWORDLONG value);
    CORINFO_CLASS_HANDLE repGetParentType(CORINFO_CLASS_HANDLE cls);

    void recIsSDArray(CORINFO_CLASS_HANDLE cls, BOOL result);
    void dmpIsSDArray(DWORDLONG key, DWORD value);
    BOOL repIsSDArray(CORINFO_CLASS_HANDLE cls);

    void recIsInSIMDModule(CORINFO_CLASS_HANDLE cls, BOOL result);
    void dmpIsInSIMDModule(DWORDLONG key, DWORD value);
    BOOL repIsInSIMDModule(CORINFO_CLASS_HANDLE cls);

    void recGetFieldClass(CORINFO_FIELD_HANDLE field, CORINFO_CLASS_HANDLE result);
    void dmpGetFieldClass(DWORDLONG key, DWORDLONG value);
    CORINFO_CLASS_HANDLE repGetFieldClass(CORINFO_FIELD_HANDLE field);

    void recGetFieldOffset(CORINFO_FIELD_HANDLE field, unsigned result);
    void dmpGetFieldOffset(DWORDLONG key, DWORD value);
    unsigned repGetFieldOffset(CORINFO_FIELD_HANDLE field);

    void recGetLazyStringLiteralHelper(CORINFO_MODULE_HANDLE handle, CorInfoHelpFunc result);
    void dmpGetLazyStringLiteralHelper(DWORDLONG key, DWORD value);
    CorInfoHelpFunc repGetLazyStringLiteralHelper(CORINFO_MODULE_HANDLE handle);

    void recGetUnBoxHelper(CORINFO_CLASS_HANDLE cls, CorInfoHelpFunc result);
    void dmpGetUnBoxHelper(DWORDLONG key, DWORD value);
    CorInfoHelpFunc repGetUnBoxHelper(CORINFO_CLASS_HANDLE cls);

    void recGetReadyToRunHelper(CORINFO_RESOLVED_TOKEN* pResolvedToken, CORINFO_LOOKUP_KIND* pGenericLookupKind, CorInfoHelpFunc id, CORINFO_CONST_LOOKUP* pLookup, bool result);
    void dmpGetReadyToRunHelper(DWORDLONG key, DWORD value);
    bool repGetReadyToRunHelper(CORINFO_RESOLVED_TOKEN* pResolvedToken, CORINFO_LOOKUP_KIND* pGenericLookupKind, CorInfoHelpFunc id, CORINFO_CONST_LOOKUP* pLookup);

    void recGetReadyToRunDelegateCtorHelper(CORINFO_RESOLVED_TOKEN* pTargetMethod, CORINFO_CLASS_HANDLE delegateType, CORINFO_CONST_LOOKUP* pLookup);
    void dmpGetReadyToRunDelegateCtorHelper(DWORDLONG key, DWORD value);
    void repGetReadyToRunDelegateCtorHelper(CORINFO_RESOLVED_TOKEN* pTargetMethod, CORINFO_CLASS_HANDLE delegateType, CORINFO_CONST_LOOKUP* pLookup);

    void recGetHelperFtn(CorInfoHelpFunc ftnNum, void **ppIndirection, void *result);
    void dmpGetHelperFtn(DWORD key, DLDL value);
    void* repGetHelperFtn(CorInfoHelpFunc ftnNum, void **ppIndirection);
    bool fndGetHelperFtn(void *functionAddress, CorInfoHelpFunc *pResult);

    void recGetJustMyCodeHandle(CORINFO_METHOD_HANDLE method, CORINFO_JUST_MY_CODE_HANDLE **ppIndirection, CORINFO_JUST_MY_CODE_HANDLE result);
    void dmpGetJustMyCodeHandle(DWORDLONG key, DLDL value);
    CORINFO_JUST_MY_CODE_HANDLE repGetJustMyCodeHandle(CORINFO_METHOD_HANDLE method, CORINFO_JUST_MY_CODE_HANDLE **ppIndirection);

    void recGetFunctionEntryPoint(CORINFO_METHOD_HANDLE ftn, CORINFO_CONST_LOOKUP *pResult, CORINFO_ACCESS_FLAGS accessFlags);
    void dmpGetFunctionEntryPoint(DLD key, DLD value);
    void repGetFunctionEntryPoint(CORINFO_METHOD_HANDLE ftn, CORINFO_CONST_LOOKUP *pResult, CORINFO_ACCESS_FLAGS accessFlags);
    bool fndGetFunctionEntryPoint(DLD value, CORINFO_METHOD_HANDLE *pResult);

    void recConstructStringLiteral(CORINFO_MODULE_HANDLE module, mdToken metaTok, void *ppValue, InfoAccessType result);
    void dmpConstructStringLiteral(DLD key, DLD value);
    InfoAccessType repConstructStringLiteral(CORINFO_MODULE_HANDLE module, mdToken metaTok, void **ppValue);

    void recEmptyStringLiteral(void **ppValue, InfoAccessType result);
    void dmpEmptyStringLiteral(DWORD key, DLD value);
    InfoAccessType repEmptyStringLiteral(void **ppValue);

    void recGetArgType(CORINFO_SIG_INFO* sig, CORINFO_ARG_LIST_HANDLE args, CORINFO_CLASS_HANDLE *vcTypeRet, CorInfoTypeWithMod result, DWORD exception);
    void dmpGetArgType(const Agnostic_GetArgType& key, const Agnostic_GetArgType_Value& value);
    CorInfoTypeWithMod repGetArgType(CORINFO_SIG_INFO* sig, CORINFO_ARG_LIST_HANDLE args, CORINFO_CLASS_HANDLE *vcTypeRet, DWORD *exception);

    void recGetArgNext(CORINFO_ARG_LIST_HANDLE args, CORINFO_ARG_LIST_HANDLE result);
    void dmpGetArgNext(DWORDLONG key, DWORDLONG value);
    CORINFO_ARG_LIST_HANDLE repGetArgNext(CORINFO_ARG_LIST_HANDLE args);

    void recGetMethodSig(CORINFO_METHOD_HANDLE ftn, CORINFO_SIG_INFO *sig, CORINFO_CLASS_HANDLE memberParent);
    void dmpGetMethodSig(DLDL key, const Agnostic_CORINFO_SIG_INFO& value);
    void repGetMethodSig(CORINFO_METHOD_HANDLE ftn, CORINFO_SIG_INFO *sig, CORINFO_CLASS_HANDLE memberParent);

    void recGetArgClass(CORINFO_SIG_INFO* sig, CORINFO_ARG_LIST_HANDLE args, CORINFO_CLASS_HANDLE result, DWORD exceptionCode);
    void dmpGetArgClass(const Agnostic_GetArgClass& key, const Agnostic_GetArgClass_Value& value);
    CORINFO_CLASS_HANDLE repGetArgClass(CORINFO_SIG_INFO* sig, CORINFO_ARG_LIST_HANDLE args, DWORD *exceptionCode);

    void recGetMethodInfo(CORINFO_METHOD_HANDLE ftn, CORINFO_METHOD_INFO *info, bool result, DWORD exceptionCode);
    void dmpGetMethodInfo(DWORDLONG key, const Agnostic_GetMethodInfo& value);
    bool repGetMethodInfo(CORINFO_METHOD_HANDLE ftn, CORINFO_METHOD_INFO *info, DWORD *exceptionCode);

    void recGetNewHelper(CORINFO_RESOLVED_TOKEN *pResolvedToken, CORINFO_METHOD_HANDLE callerHandle, CorInfoHelpFunc result);
    void dmpGetNewHelper(const Agnostic_GetNewHelper& key, DWORD value);
    CorInfoHelpFunc repGetNewHelper(CORINFO_RESOLVED_TOKEN *pResolvedToken, CORINFO_METHOD_HANDLE callerHandle);

    void recEmbedGenericHandle(CORINFO_RESOLVED_TOKEN *pResolvedToken, BOOL fEmbedParent, CORINFO_GENERICHANDLE_RESULT *pResult);
    void dmpEmbedGenericHandle(const Agnostic_EmbedGenericHandle& key, const Agnostic_CORINFO_GENERICHANDLE_RESULT& value);
    void repEmbedGenericHandle(CORINFO_RESOLVED_TOKEN *pResolvedToken, BOOL fEmbedParent, CORINFO_GENERICHANDLE_RESULT *pResult);

    void recGetEHinfo(CORINFO_METHOD_HANDLE ftn, unsigned EHnumber, CORINFO_EH_CLAUSE *clause);
    void dmpGetEHinfo(DLD key, const Agnostic_CORINFO_EH_CLAUSE& value);
    void repGetEHinfo(CORINFO_METHOD_HANDLE ftn, unsigned EHnumber, CORINFO_EH_CLAUSE *clause);

    void recGetMethodVTableOffset(CORINFO_METHOD_HANDLE method, unsigned *offsetOfIndirection, unsigned* offsetAfterIndirection);
    void dmpGetMethodVTableOffset(DWORDLONG key, DD value);
    void repGetMethodVTableOffset(CORINFO_METHOD_HANDLE method, unsigned *offsetOfIndirection, unsigned* offsetAfterIndirection);

    void recGetTokenTypeAsHandle(CORINFO_RESOLVED_TOKEN * pResolvedToken, CORINFO_CLASS_HANDLE result);
    void dmpGetTokenTypeAsHandle(const Agnostic_CORINFO_RESOLVED_TOKEN& key, DWORDLONG value);
    CORINFO_CLASS_HANDLE repGetTokenTypeAsHandle(CORINFO_RESOLVED_TOKEN * pResolvedToken);

    void recGetFieldInfo(CORINFO_RESOLVED_TOKEN *pResolvedToken, CORINFO_METHOD_HANDLE callerHandle, CORINFO_ACCESS_FLAGS flags,
        CORINFO_FIELD_INFO *pResult);
    void dmpGetFieldInfo(const Agnostic_GetFieldInfo& key, const Agnostic_CORINFO_FIELD_INFO& value);
    void repGetFieldInfo(CORINFO_RESOLVED_TOKEN *pResolvedToken, CORINFO_METHOD_HANDLE callerHandle, CORINFO_ACCESS_FLAGS flags,
        CORINFO_FIELD_INFO *pResult);

    void recEmbedMethodHandle(CORINFO_METHOD_HANDLE handle, void **ppIndirection, CORINFO_METHOD_HANDLE result);
    void dmpEmbedMethodHandle(DWORDLONG key, DLDL value);
    CORINFO_METHOD_HANDLE repEmbedMethodHandle(CORINFO_METHOD_HANDLE handle, void **ppIndirection);

    void recGetFieldAddress(CORINFO_FIELD_HANDLE field, void **ppIndirection, void *result, CorInfoType cit);
    void dmpGetFieldAddress(DWORDLONG key, const Agnostic_GetFieldAddress& value);
    void* repGetFieldAddress(CORINFO_FIELD_HANDLE field, void **ppIndirection);

    void recGetClassGClayout(CORINFO_CLASS_HANDLE cls, BYTE *gcPtrs, unsigned len, unsigned result);
    void dmpGetClassGClayout(DWORDLONG key, const Agnostic_GetClassGClayout& value);
    unsigned repGetClassGClayout(CORINFO_CLASS_HANDLE cls, BYTE *gcPtrs);

    void recGetClassAlignmentRequirement(CORINFO_CLASS_HANDLE cls, BOOL fDoubleAlignHint, unsigned result);
    void dmpGetClassAlignmentRequirement(DLD key, DWORD value);
    unsigned repGetClassAlignmentRequirement(CORINFO_CLASS_HANDLE cls, BOOL fDoubleAlignHint);

    void recCanAccessClass(CORINFO_RESOLVED_TOKEN *pResolvedToken, CORINFO_METHOD_HANDLE callerHandle,
        CORINFO_HELPER_DESC *pAccessHelper, CorInfoIsAccessAllowedResult  result);
    void dmpCanAccessClass(const Agnostic_CanAccessClassIn& key, const Agnostic_CanAccessClassOut& value);
    CorInfoIsAccessAllowedResult repCanAccessClass(CORINFO_RESOLVED_TOKEN *pResolvedToken, CORINFO_METHOD_HANDLE callerHandle,
        CORINFO_HELPER_DESC *pAccessHelper);

    void recGetCastingHelper(CORINFO_RESOLVED_TOKEN *pResolvedToken, bool fThrowing, CorInfoHelpFunc result);
    void dmpGetCastingHelper(const Agnostic_GetCastingHelper& key, DWORD value);
    CorInfoHelpFunc repGetCastingHelper(CORINFO_RESOLVED_TOKEN *pResolvedToken, bool fThrowing);

    void recEmbedModuleHandle(CORINFO_MODULE_HANDLE handle, void **ppIndirection, CORINFO_MODULE_HANDLE result);
    void dmpEmbedModuleHandle(DWORDLONG key, DLDL value);
    CORINFO_MODULE_HANDLE repEmbedModuleHandle(CORINFO_MODULE_HANDLE handle, void **ppIndirection);

    void recEmbedClassHandle(CORINFO_CLASS_HANDLE handle, void **ppIndirection, CORINFO_CLASS_HANDLE result);
    void dmpEmbedClassHandle(DWORDLONG key, DLDL value);
    CORINFO_CLASS_HANDLE repEmbedClassHandle(CORINFO_CLASS_HANDLE handle, void **ppIndirection);

    void recPInvokeMarshalingRequired(CORINFO_METHOD_HANDLE method, CORINFO_SIG_INFO* callSiteSig, BOOL result);
    void dmpPInvokeMarshalingRequired(const Agnostic_PInvokeMarshalingRequired& key, DWORD value);
    BOOL repPInvokeMarshalingRequired(CORINFO_METHOD_HANDLE method, CORINFO_SIG_INFO* callSiteSig);

    void recFindSig(CORINFO_MODULE_HANDLE module, unsigned sigTOK, CORINFO_CONTEXT_HANDLE context, CORINFO_SIG_INFO *sig);
    void dmpFindSig(const Agnostic_FindSig& key, const Agnostic_CORINFO_SIG_INFO& value);
    void repFindSig(CORINFO_MODULE_HANDLE module, unsigned sigTOK, CORINFO_CONTEXT_HANDLE context, CORINFO_SIG_INFO *sig);

    void recGetEEInfo(CORINFO_EE_INFO *pEEInfoOut);
    void dmpGetEEInfo(DWORD key, const Agnostic_CORINFO_EE_INFO& value);
    void repGetEEInfo(CORINFO_EE_INFO *pEEInfoOut);

    void recGetGSCookie(GSCookie *pCookieVal, GSCookie **ppCookieVal);
    void dmpGetGSCookie(DWORD key, DLDL value);
    void repGetGSCookie(GSCookie *pCookieVal, GSCookie **ppCookieVal);

    void recGetClassModuleIdForStatics(CORINFO_CLASS_HANDLE cls, CORINFO_MODULE_HANDLE *pModule, void **ppIndirection, size_t result);
    void dmpGetClassModuleIdForStatics(DWORDLONG key, const Agnostic_GetClassModuleIdForStatics& value);
    size_t repGetClassModuleIdForStatics(CORINFO_CLASS_HANDLE cls, CORINFO_MODULE_HANDLE *pModule, void **ppIndirection);

    void recGetThreadTLSIndex(void **ppIndirection, DWORD result);
    void dmpGetThreadTLSIndex(DWORD key, DLD value);
    DWORD repGetThreadTLSIndex(void **ppIndirection);

    void recGetInlinedCallFrameVptr(void **ppIndirection, const void * result);
    void dmpGetInlinedCallFrameVptr(DWORD key, DLDL value);
    const void * repGetInlinedCallFrameVptr(void **ppIndirection);

    void recGetAddrOfCaptureThreadGlobal(void **ppIndirection, LONG * result);
    void dmpGetAddrOfCaptureThreadGlobal(DWORD key, DLDL value);
    LONG * repGetAddrOfCaptureThreadGlobal(void **ppIndirection);

    void recGetClassDomainID(CORINFO_CLASS_HANDLE cls, void **ppIndirection, unsigned result);
    void dmpGetClassDomainID(DWORDLONG key, DLD value);
    unsigned repGetClassDomainID(CORINFO_CLASS_HANDLE cls, void **ppIndirection);

    void recGetLocationOfThisType(CORINFO_METHOD_HANDLE context, CORINFO_LOOKUP_KIND *result);
    void dmpGetLocationOfThisType(DWORDLONG key, const Agnostic_CORINFO_LOOKUP_KIND& value);
    CORINFO_LOOKUP_KIND repGetLocationOfThisType(CORINFO_METHOD_HANDLE context);

    void recGetDelegateCtor(CORINFO_METHOD_HANDLE  methHnd, CORINFO_CLASS_HANDLE clsHnd,
        CORINFO_METHOD_HANDLE  targetMethodHnd, DelegateCtorArgs *pCtorData, CORINFO_METHOD_HANDLE result);
    void dmpGetDelegateCtor(const Agnostic_GetDelegateCtorIn& key, const Agnostic_GetDelegateCtorOut& value);
    CORINFO_METHOD_HANDLE repGetDelegateCtor(CORINFO_METHOD_HANDLE  methHnd, CORINFO_CLASS_HANDLE clsHnd,
        CORINFO_METHOD_HANDLE  targetMethodHnd, DelegateCtorArgs *pCtorData);

    void recGetFunctionFixedEntryPoint(CORINFO_METHOD_HANDLE ftn, CORINFO_CONST_LOOKUP *pResult);
    void dmpGetFunctionFixedEntryPoint(DWORDLONG key, const Agnostic_CORINFO_CONST_LOOKUP& value);
    void repGetFunctionFixedEntryPoint(CORINFO_METHOD_HANDLE ftn, CORINFO_CONST_LOOKUP *pResult);

    void recGetFieldInClass(CORINFO_CLASS_HANDLE clsHnd, INT num, CORINFO_FIELD_HANDLE result);
    void dmpGetFieldInClass(DLD key, DWORDLONG value);
    CORINFO_FIELD_HANDLE repGetFieldInClass(CORINFO_CLASS_HANDLE clsHnd, INT num);

    void recGetFieldType(CORINFO_FIELD_HANDLE field, CORINFO_CLASS_HANDLE *structType, CORINFO_CLASS_HANDLE memberParent, CorInfoType result);
    void dmpGetFieldType(DLDL key, DLD value);
    CorInfoType repGetFieldType(CORINFO_FIELD_HANDLE field, CORINFO_CLASS_HANDLE *structType, CORINFO_CLASS_HANDLE memberParent);

    void recGetFieldName(CORINFO_FIELD_HANDLE ftn, const char **moduleName, const char* result);
    void dmpGetFieldName(DWORDLONG key, DD value);
    const char* repGetFieldName(CORINFO_FIELD_HANDLE ftn, const char **moduleName);

    void recCanInlineTypeCheckWithObjectVTable(CORINFO_CLASS_HANDLE cls, BOOL result);
    void dmpCanInlineTypeCheckWithObjectVTable(DWORDLONG key, DWORD value);
    BOOL repCanInlineTypeCheckWithObjectVTable(CORINFO_CLASS_HANDLE cls);

    void recSatisfiesMethodConstraints(CORINFO_CLASS_HANDLE parent, CORINFO_METHOD_HANDLE method, BOOL result);
    void dmpSatisfiesMethodConstraints(DLDL key, DWORD value);
    BOOL repSatisfiesMethodConstraints(CORINFO_CLASS_HANDLE parent, CORINFO_METHOD_HANDLE method);

    void recInitConstraintsForVerification(CORINFO_METHOD_HANDLE method, BOOL *pfHasCircularClassConstraints,
        BOOL *pfHasCircularMethodConstraint);
    void dmpInitConstraintsForVerification(DWORDLONG key, DD value);
    void repInitConstraintsForVerification(CORINFO_METHOD_HANDLE method, BOOL *pfHasCircularClassConstraints,
        BOOL *pfHasCircularMethodConstraint);

    void recIsValidStringRef(CORINFO_MODULE_HANDLE module, unsigned metaTOK, BOOL result);
    void dmpIsValidStringRef(DLD key, DWORD value);
    BOOL repIsValidStringRef(CORINFO_MODULE_HANDLE module, unsigned metaTOK);

    void recGetHelperName(CorInfoHelpFunc funcNum, const char* result);
    void dmpGetHelperName(DWORD key, DWORD value);
    const char* repGetHelperName(CorInfoHelpFunc funcNum);

    void recCanCast(CORINFO_CLASS_HANDLE child, CORINFO_CLASS_HANDLE parent, BOOL result);
    void dmpCanCast(DLDL key, DWORD value);
    BOOL repCanCast(CORINFO_CLASS_HANDLE child, CORINFO_CLASS_HANDLE parent);

    void recGetChildType(CORINFO_CLASS_HANDLE clsHnd, CORINFO_CLASS_HANDLE *clsRet, CorInfoType result);
    void dmpGetChildType(DWORDLONG key, DLD value);
    CorInfoType repGetChildType(CORINFO_CLASS_HANDLE clsHnd, CORINFO_CLASS_HANDLE *clsRet);

    void recGetArrayInitializationData(CORINFO_FIELD_HANDLE field, DWORD size, void *result);
    void dmpGetArrayInitializationData(DLD key, DWORDLONG value);
    void *repGetArrayInitializationData(CORINFO_FIELD_HANDLE field, DWORD size);

    void recFilterException(struct _EXCEPTION_POINTERS *pExceptionPointers, int result);
    void dmpFilterException(DWORD key, DWORD value);
    int repFilterException(struct _EXCEPTION_POINTERS *pExceptionPointers);

    void recHandleException(struct _EXCEPTION_POINTERS *pExceptionPointers);
    void dmpHandleException(DWORD key, DWORD value);

    void recGetAddressOfPInvokeFixup(CORINFO_METHOD_HANDLE method, void **ppIndirection, void* result);
    void dmpGetAddressOfPInvokeFixup(DWORDLONG key, DLDL value);
    void* repGetAddressOfPInvokeFixup(CORINFO_METHOD_HANDLE method, void **ppIndirection);

    void recGetAddressOfPInvokeTarget(CORINFO_METHOD_HANDLE method, CORINFO_CONST_LOOKUP *pLookup);
    void dmpGetAddressOfPInvokeTarget(DWORDLONG key, DLD value);
    void repGetAddressOfPInvokeTarget(CORINFO_METHOD_HANDLE method, CORINFO_CONST_LOOKUP *pLookup);

    void recSatisfiesClassConstraints(CORINFO_CLASS_HANDLE cls, BOOL result);
    void dmpSatisfiesClassConstraints(DWORDLONG key, DWORD value);
    BOOL repSatisfiesClassConstraints(CORINFO_CLASS_HANDLE cls);

    void recGetMethodHash(CORINFO_METHOD_HANDLE ftn, unsigned result);
    void dmpGetMethodHash(DWORDLONG key, DWORD value);
    unsigned repGetMethodHash(CORINFO_METHOD_HANDLE ftn);

    void recCanTailCall(CORINFO_METHOD_HANDLE callerHnd, CORINFO_METHOD_HANDLE declaredCalleeHnd, CORINFO_METHOD_HANDLE exactCalleeHnd,
        bool fIsTailPrefix, bool result);
    void dmpCanTailCall(const Agnostic_CanTailCall& key, DWORD value);
    bool repCanTailCall(CORINFO_METHOD_HANDLE callerHnd, CORINFO_METHOD_HANDLE declaredCalleeHnd, CORINFO_METHOD_HANDLE exactCalleeHnd,
        bool fIsTailPrefix);

    void recIsCompatibleDelegate(CORINFO_CLASS_HANDLE objCls, CORINFO_CLASS_HANDLE methodParentCls,
        CORINFO_METHOD_HANDLE method, CORINFO_CLASS_HANDLE delegateCls, BOOL *pfIsOpenDelegate, BOOL result);
    void dmpIsCompatibleDelegate(const Agnostic_IsCompatibleDelegate& key, DD value);
    BOOL repIsCompatibleDelegate(CORINFO_CLASS_HANDLE objCls, CORINFO_CLASS_HANDLE methodParentCls,
        CORINFO_METHOD_HANDLE method, CORINFO_CLASS_HANDLE delegateCls, BOOL *pfIsOpenDelegate);

    void recIsDelegateCreationAllowed(CORINFO_CLASS_HANDLE delegateHnd, CORINFO_METHOD_HANDLE calleeHnd, BOOL result);
    void dmpIsDelegateCreationAllowed(DLDL key, DWORD value);
    BOOL repIsDelegateCreationAllowed(CORINFO_CLASS_HANDLE delegateHnd, CORINFO_METHOD_HANDLE calleeHnd);

    void recCanSkipMethodVerification(CORINFO_METHOD_HANDLE ftnHandle, BOOL skip, CorInfoCanSkipVerificationResult result);
    void dmpCanSkipMethodVerification(DLD key, DWORD value);
    CorInfoCanSkipVerificationResult repCanSkipMethodVerification(CORINFO_METHOD_HANDLE ftnHandle, BOOL skip);

    void recFindCallSiteSig(CORINFO_MODULE_HANDLE module, unsigned methTOK, CORINFO_CONTEXT_HANDLE context, CORINFO_SIG_INFO *sig);
    void dmpFindCallSiteSig(const Agnostic_FindCallSiteSig& key, const Agnostic_CORINFO_SIG_INFO& value);
    void repFindCallSiteSig(CORINFO_MODULE_HANDLE module, unsigned methTOK, CORINFO_CONTEXT_HANDLE context, CORINFO_SIG_INFO *sig);

    void recShouldEnforceCallvirtRestriction(CORINFO_MODULE_HANDLE scope, BOOL result);
    void dmpShouldEnforceCallvirtRestriction(DWORDLONG key, DWORD value);
    BOOL repShouldEnforceCallvirtRestriction(CORINFO_MODULE_HANDLE scope);

    void recGetMethodSync(CORINFO_METHOD_HANDLE ftn, void **ppIndirection, void* result);
    void dmpGetMethodSync(DWORDLONG key, DLDL value);
    void* repGetMethodSync(CORINFO_METHOD_HANDLE ftn, void **ppIndirection);

    void  recGetVarArgsHandle(CORINFO_SIG_INFO *pSig, void **ppIndirection, CORINFO_VARARGS_HANDLE result);
    void dmpGetVarArgsHandle(const Agnostic_CORINFO_SIG_INFO& key, DLDL value);
    CORINFO_VARARGS_HANDLE repGetVarArgsHandle(CORINFO_SIG_INFO *pSig, void **ppIndirection);

    void recCanGetVarArgsHandle(CORINFO_SIG_INFO *pSig, bool result);
    void dmpCanGetVarArgsHandle(const Agnostic_CORINFO_SIG_INFO& key, DWORD value);
    bool repCanGetVarArgsHandle(CORINFO_SIG_INFO *pSig);

    void recGetFieldThreadLocalStoreID(CORINFO_FIELD_HANDLE field, void **ppIndirection, DWORD result);
    void dmpGetFieldThreadLocalStoreID(DWORDLONG key, DLD value);
    DWORD repGetFieldThreadLocalStoreID(CORINFO_FIELD_HANDLE field, void **ppIndirection);

    void recGetBBProfileData(CORINFO_METHOD_HANDLE ftnHnd, ULONG *count, ICorJitInfo::ProfileBuffer **profileBuffer,
        ULONG *numRuns, HRESULT  result);
    void dmpGetBBProfileData(DWORDLONG key, const Agnostic_GetBBProfileData& value);
    HRESULT repGetBBProfileData(CORINFO_METHOD_HANDLE ftnHnd, ULONG *count, ICorJitInfo::ProfileBuffer **profileBuffer,
        ULONG *numRuns);

    void recMergeClasses(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2, CORINFO_CLASS_HANDLE result);
    void dmpMergeClasses(DLDL key, DWORDLONG value);
    CORINFO_CLASS_HANDLE repMergeClasses(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2);

    void recGetCookieForPInvokeCalliSig(CORINFO_SIG_INFO* szMetaSig, void ** ppIndirection, LPVOID result);
    void dmpGetCookieForPInvokeCalliSig(const Agnostic_CORINFO_SIG_INFO& key, DLDL value);
    LPVOID repGetCookieForPInvokeCalliSig(CORINFO_SIG_INFO* szMetaSig, void ** ppIndirection);

    void recCanGetCookieForPInvokeCalliSig(CORINFO_SIG_INFO* szMetaSig, bool result);
    void dmpCanGetCookieForPInvokeCalliSig(const Agnostic_CORINFO_SIG_INFO& key, DWORD value);
    bool repCanGetCookieForPInvokeCalliSig(CORINFO_SIG_INFO* szMetaSig);

    void recCanAccessFamily(CORINFO_METHOD_HANDLE hCaller, CORINFO_CLASS_HANDLE hInstanceType, BOOL result);
    void dmpCanAccessFamily(DLDL key, DWORD value);
    BOOL repCanAccessFamily(CORINFO_METHOD_HANDLE hCaller, CORINFO_CLASS_HANDLE hInstanceType);

    void recErrorList(const char *error);
    void dmpErrorList(DWORD key, DWORD value);

    void recGetProfilingHandle(BOOL *pbHookFunction, void **pProfilerHandle, BOOL *pbIndirectedHandles);
    void dmpGetProfilingHandle(DWORD key, const Agnostic_GetProfilingHandle& value);
    void repGetProfilingHandle(BOOL *pbHookFunction, void **pProfilerHandle, BOOL *pbIndirectedHandles);

    void recEmbedFieldHandle(CORINFO_FIELD_HANDLE handle, void **ppIndirection, CORINFO_FIELD_HANDLE result);
    void dmpEmbedFieldHandle(DWORDLONG key, DLDL value);
    CORINFO_FIELD_HANDLE repEmbedFieldHandle(CORINFO_FIELD_HANDLE handle, void **ppIndirection);

    void recAreTypesEquivalent(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2, BOOL result);
    void dmpAreTypesEquivalent(DLDL key, DWORD value);
    BOOL repAreTypesEquivalent(CORINFO_CLASS_HANDLE cls1, CORINFO_CLASS_HANDLE cls2);

    void recFindNameOfToken(CORINFO_MODULE_HANDLE module, mdToken metaTOK, char * szFQName, size_t FQNameCapacity, size_t result);
    void dmpFindNameOfToken(DLD key, DLD value);
    size_t repFindNameOfToken(CORINFO_MODULE_HANDLE module, mdToken metaTOK, char * szFQName, size_t FQNameCapacity);

    void recGetSystemVAmd64PassStructInRegisterDescriptor(CORINFO_CLASS_HANDLE structHnd, SYSTEMV_AMD64_CORINFO_STRUCT_REG_PASSING_DESCRIPTOR* structPassInRegDescPtr, bool result);
    void dmpGetSystemVAmd64PassStructInRegisterDescriptor(DWORDLONG key, const Agnostic_GetSystemVAmd64PassStructInRegisterDescriptor& value);
    bool repGetSystemVAmd64PassStructInRegisterDescriptor(CORINFO_CLASS_HANDLE structHnd, SYSTEMV_AMD64_CORINFO_STRUCT_REG_PASSING_DESCRIPTOR* structPassInRegDescPtr);

    void recGetRelocTypeHint(void * target, WORD result);
    void dmpGetRelocTypeHint(DWORDLONG key, DWORD value);
    WORD repGetRelocTypeHint(void * target);

    void recIsWriteBarrierHelperRequired(CORINFO_FIELD_HANDLE field, bool result);
    void dmpIsWriteBarrierHelperRequired(DWORDLONG key, DWORD value);
    bool repIsWriteBarrierHelperRequired(CORINFO_FIELD_HANDLE field);

    void recIsValidToken(CORINFO_MODULE_HANDLE module, unsigned metaTOK, BOOL result);
    void dmpIsValidToken(DLD key, DWORD value);
    BOOL repIsValidToken(CORINFO_MODULE_HANDLE module, unsigned metaTOK);

    void recGetClassName(CORINFO_CLASS_HANDLE cls, const char* result);
    void dmpGetClassName(DWORDLONG key, DWORD value);
    const char* repGetClassName(CORINFO_CLASS_HANDLE cls);

    void recAppendClassName(CORINFO_CLASS_HANDLE cls, BOOL fNamespace, BOOL fFullInst, BOOL fAssembly, const WCHAR* result);
    void dmpAppendClassName(const Agnostic_AppendClassName& key, DWORD value);
    const WCHAR* repAppendClassName(CORINFO_CLASS_HANDLE cls, BOOL fNamespace, BOOL fFullInst, BOOL fAssembly);

    void recGetTailCallCopyArgsThunk(CORINFO_SIG_INFO *pSig, CorInfoHelperTailCallSpecialHandling flags, void* result);
    void dmpGetTailCallCopyArgsThunk(const Agnostic_GetTailCallCopyArgsThunk& key, DWORDLONG value);
    void* repGetTailCallCopyArgsThunk(CORINFO_SIG_INFO *pSig, CorInfoHelperTailCallSpecialHandling flags);

    void recGetMethodDefFromMethod(CORINFO_METHOD_HANDLE hMethod, mdMethodDef result);
    void dmpGetMethodDefFromMethod(DWORDLONG key, DWORD value);
    mdMethodDef repGetMethodDefFromMethod(CORINFO_METHOD_HANDLE hMethod);

    void recCheckMethodModifier(CORINFO_METHOD_HANDLE hMethod, LPCSTR modifier, BOOL fOptional, BOOL result);
    void dmpCheckMethodModifier(const Agnostic_CheckMethodModifier& key, DWORD value);
    BOOL repCheckMethodModifier(CORINFO_METHOD_HANDLE hMethod, LPCSTR modifier, BOOL fOptional);

    void recGetPInvokeUnmanagedTarget(CORINFO_METHOD_HANDLE method, void **ppIndirection, void* result);
    void dmpGetPInvokeUnmanagedTarget(DWORDLONG key, DLDL value);
    void* repGetPInvokeUnmanagedTarget(CORINFO_METHOD_HANDLE method, void **ppIndirection);

    void recGetArrayRank(CORINFO_CLASS_HANDLE cls, unsigned result);
    void dmpGetArrayRank(DWORDLONG key, DWORD value);
    unsigned repGetArrayRank(CORINFO_CLASS_HANDLE cls);

    void recIsFieldStatic(CORINFO_FIELD_HANDLE fhld, bool result);
    void dmpIsFieldStatic(DWORDLONG key, DWORD value);
    bool repIsFieldStatic(CORINFO_FIELD_HANDLE fhld);

    void recGetIntConfigValue(const wchar_t *name, int defaultValue, int result);
    void dmpGetIntConfigValue(const Agnostic_ConfigIntInfo& key, int value);
    int  repGetIntConfigValue(const wchar_t *name, int defaultValue);

    void recGetStringConfigValue(const wchar_t *name, const wchar_t *result);
    void dmpGetStringConfigValue(DWORD nameIndex, DWORD result);
    const wchar_t *repGetStringConfigValue(const wchar_t *name);

    CompileResult *cr;
    CompileResult *originalCR;
    int index;

private:

    #define LWM(map,key,value) LightWeightMap<key, value>* map;
    #define DENSELWM(map,value) DenseLightWeightMap<value>* map;
    #include "lwmlist.h"

};


// ********************* Please keep this up-to-date to ease adding more ***************
// Highest packet number: 158
// *************************************************************************************
enum mcPackets
{
    Packet_AppendClassName = 149,  // Added 8/6/2014 - needed for SIMD
    Packet_AreTypesEquivalent = 1,
    Packet_AsCorInfoType = 2,
    Packet_CanAccessClass = 3,
    Packet_CanAccessFamily = 4,
    Packet_CanCast = 5,
    Retired8 = 6,
    Packet_GetLazyStringLiteralHelper = 147,  //Added 12/20/2013 - as a replacement for CanEmbedModuleHandleForHelper
    Packet_CanGetCookieForPInvokeCalliSig = 7,
    Packet_CanGetVarArgsHandle = 8,
    Packet_CanInline = 9,
    Packet_CanInlineTypeCheckWithObjectVTable = 10,
    Packet_CanSkipMethodVerification = 11,
    Packet_CanTailCall = 12,
    Retired4 = 13,
    Packet_CheckMethodModifier = 142, //retired as 13 on 2013/07/04
    Retired3 = 14,
    Retired5 = 141, //retired as 14 on 2013/07/03
    Packet_CompileMethod = 143, //retired as 141 on 2013/07/09
    Packet_ConstructStringLiteral = 15,
    Packet_EmbedClassHandle = 16,
    Packet_EmbedFieldHandle = 17,
    Packet_EmbedGenericHandle = 18,
    Packet_EmbedMethodHandle = 19,
    Packet_EmbedModuleHandle = 20,
    Packet_EmptyStringLiteral = 21,
    Packet_Environment = 136, //Added 4/3/2013
    Packet_ErrorList = 22,
    Packet_FilterException = 134,
    Packet_FindCallSiteSig = 23,
    Retired7 = 24,
    Packet_FindNameOfToken = 145,//Added 7/19/2013 - adjusted members to proper types
    Packet_GetSystemVAmd64PassStructInRegisterDescriptor = 156, //Added 2/17/2016
    Packet_FindSig = 25,
    Packet_GetAddressOfPInvokeFixup = 26,
    Packet_GetAddressOfPInvokeTarget = 153, //Added 2/3/2016
    Packet_GetAddrOfCaptureThreadGlobal = 27,
    Retired1 = 28,
    Packet_GetArgClass = 139, //retired as 28 on 2013/07/03
    Packet_GetArgNext = 29,
    Retired2 = 30,
    Packet_GetArgType = 140, //retired as 30 on 2013/07/03
    Packet_GetArrayInitializationData = 31,
    Packet_GetArrayRank = 32,
    Packet_GetBBProfileData = 33,
    Packet_GetBoundaries = 34,
    Packet_GetBoxHelper = 35,
    Packet_GetBuiltinClass = 36,
    Packet_GetCallInfo = 37,
    Packet_GetCastingHelper = 38,
    Packet_GetChildType = 39,
    Packet_GetClassAlignmentRequirement = 40,
    Packet_GetClassAttribs = 41,
    Packet_GetClassDomainID = 42,
    Packet_GetClassGClayout = 43,
    Packet_GetClassModuleIdForStatics = 44,
    Packet_GetClassName = 45,
    Packet_GetClassNumInstanceFields = 46,
    Packet_GetClassSize = 47,
    Packet_GetIntConfigValue = 151, //Added 2/12/2015
    Packet_GetStringConfigValue = 152, // Added 2/12/2015
    Packet_GetCookieForPInvokeCalliSig = 48,
    Packet_GetDelegateCtor = 49,
    Packet_GetEEInfo = 50,
    Packet_GetEHinfo = 51,
    Packet_GetFieldAddress = 52,
    Packet_GetFieldClass = 53,
    Packet_GetFieldInClass = 54,
    Packet_GetFieldInfo = 55,
    Packet_GetFieldName = 56,
    Packet_GetFieldOffset = 57,
    Packet_GetFieldThreadLocalStoreID = 58,
    Packet_GetFieldType = 59,
    Packet_GetFunctionEntryPoint = 60,
    Packet_GetFunctionFixedEntryPoint = 61,
    Packet_GetGSCookie = 62,
    Packet_GetHelperFtn = 63,
    Packet_GetHelperName = 64,
    Packet_GetInlinedCallFrameVptr = 65,
    Packet_GetIntrinsicID = 66,
    Packet_GetJitFlags = 154, //Added 2/3/2016
    Packet_GetJitTimeLogFilename = 67,
    Packet_GetJustMyCodeHandle = 68,
    Packet_GetLocationOfThisType = 69,
    Packet_GetMethodAttribs = 70,
    Packet_GetMethodClass = 71,
    Packet_GetMethodDefFromMethod = 72,
    Packet_GetMethodHash = 73,
    Packet_GetMethodInfo = 74,
    Packet_GetMethodName = 75,
    Packet_GetMethodSig = 76,
    Packet_GetMethodSync = 77,
    Packet_GetMethodVTableOffset = 78,
    Packet_GetNewArrHelper = 79,
    Packet_GetNewHelper = 80,
    Packet_GetParentType = 81,
    Packet_GetPInvokeUnmanagedTarget = 82,
    Packet_GetProfilingHandle = 83,
    Packet_GetRelocTypeHint = 84,
    Packet_GetSecurityPrologHelper = 85,
    Packet_GetSharedCCtorHelper = 86,
    Packet_GetTailCallCopyArgsThunk = 87,
    Packet_GetThreadTLSIndex = 88,
    Packet_GetTokenTypeAsHandle = 89,
    Packet_GetTypeForBox = 90,
    Packet_GetTypeForPrimitiveValueClass = 91,
    Packet_GetUnBoxHelper = 92,
    Packet_GetReadyToRunHelper = 150, //Added 10/10/2014
    Packet_GetReadyToRunDelegateCtorHelper = 157, //Added 3/30/2016
    Packet_GetUnmanagedCallConv = 94,
    Packet_GetVarArgsHandle = 95,
    Packet_GetVars = 96,
    Packet_HandleException = 135,
    Packet_InitClass = 97,
    Packet_InitConstraintsForVerification = 98,
    Packet_IsCompatibleDelegate = 99,
    Packet_IsDelegateCreationAllowed = 155,
    Packet_IsFieldStatic = 137,//Added 4/9/2013 - needed for 4.5.1
    Packet_IsInSIMDModule = 148,// Added 6/18/2014 - SIMD support
    Packet_IsInstantiationOfVerifiedGeneric = 100,
    Packet_IsSDArray = 101,
    Packet_IsStructRequiringStackAllocRetBuf = 102,
    Packet_IsValidStringRef = 103,
    Retired6 = 104,
    Packet_IsValidToken = 144,//Added 7/19/2013 - adjusted members to proper types
    Packet_IsValueClass = 105,
    Packet_IsWriteBarrierHelperRequired = 106,
    Packet_MergeClasses = 107,
    Packet_PInvokeMarshalingRequired = 108,
    Packet_ResolveToken = 109,
    Packet_TryResolveToken = 158, //Added 4/26/2016
    Packet_SatisfiesClassConstraints = 110,
    Packet_SatisfiesMethodConstraints = 111,
    Packet_ShouldEnforceCallvirtRestriction = 112,

    PacketCR_AddressMap = 113,
    PacketCR_AllocBBProfileBuffer = 131,
    PacketCR_AllocGCInfo = 114,
    PacketCR_AllocMem = 115,
    PacketCR_AllocUnwindInfo = 132,
    PacketCR_AssertLog = 138, //Added 6/10/2013 - added to nicely support ilgen
    PacketCR_CallLog = 116,
    PacketCR_ClassMustBeLoadedBeforeCodeIsRun = 117,
    PacketCR_CompileMethod = 118,
    PacketCR_MessageLog = 119,
    PacketCR_MethodMustBeLoadedBeforeCodeIsRun = 120,
    PacketCR_ProcessName = 121,
    PacketCR_RecordRelocation = 122,
    PacketCR_ReportFatalError = 123,
    PacketCR_ReportInliningDecision = 124,
    PacketCR_ReportTailCallDecision = 125,
    PacketCR_ReserveUnwindInfo = 133,
    PacketCR_SetBoundaries = 126,
    PacketCR_SetEHcount = 127,
    PacketCR_SetEHinfo = 128,
    PacketCR_SetMethodAttribs = 129,
    PacketCR_SetVars = 130,
    PacketCR_RecordCallSite = 146,  //Added 10/28/2013 - to support indirect calls
};

#endif
