# Copyright (C) 2015-2019 MongoDB, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

module Mongo
  class Cursor
    module Builder

      # Encapsulates behavior around generating an OP_KILL_CURSORS specification.
      #
      # @since 2.2.0
      class OpKillCursors
        extend Forwardable

        # @return [ Cursor ] cursor The cursor.
        attr_reader :cursor

        def_delegators :@cursor, :collection_name, :database

        # Create the new builder.
        #
        # @example Create the builder.
        #   OpKillCursors.new(cursor)
        #
        # @param [ Cursor ] cursor The cursor.
        #
        # @since 2.2.0
        def initialize(cursor)
          @cursor = cursor
        end

        # Get the specification.
        #
        # @example Get the specification.
        #   op_kill_cursors.specification
        #
        # @return [ Hash ] The specification.
        #
        # @since 2.2.0
        def specification
          {
            coll_name: collection_name,
            db_name: database.name,
            cursor_ids: [ BSON::Int64.new(cursor.id) ],
          }
        end

        class << self

          # Update a specification's list of cursor ids.
          #
          # @example Update a specification's list of cursor ids.
          #   OpKillCursors.update_cursors(spec, ids)
          #
          # @return [ Hash ] The specification.
          # @return [ Array<Integer> ] The ids to update with.
          #
          # @since 2.3.0
          def update_cursors(spec, ids)
            # Ruby 2.5+ can & BSON::Int64 instances.
            # Ruby 2.4 and earlier cannot.
            # Convert stored ids to Ruby integers for compatibility with
            # older Rubies.
            ids = get_cursors_list(spec) & ids
            ids = ids.map do |cursor_id|
              BSON::Int64.new(cursor_id)
            end
            spec.merge!(cursor_ids: ids)
          end

          # Get the list of cursor ids from a spec generated by this Builder.
          #
          # @example Get the list of cursor ids.
          #   OpKillCursors.cursors(spec)
          #
          # @return [ Array<Integer> ] The cursor ids.
          #
          # @since 2.3.0
          def get_cursors_list(spec)
            spec[:cursor_ids].map do |value|
              if value.respond_to?(:value)
                # bson-ruby >= 4.6.0
                value.value
              else
                value.instance_variable_get('@integer')
              end
            end
          end
        end
      end
    end
  end
end
