//%LICENSE////////////////////////////////////////////////////////////////
//
// Licensed to The Open Group (TOG) under one or more contributor license
// agreements.  Refer to the OpenPegasusNOTICE.txt file distributed with
// this work for additional information regarding copyright ownership.
// Each contributor licenses this file to you under the OpenPegasus Open
// Source License; you may not use this file except in compliance with the
// License.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
// CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
// TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
// SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
//////////////////////////////////////////////////////////////////////////
//
//%/////////////////////////////////////////////////////////////////////////////

#ifndef Pegasus_Magic_h
#define Pegasus_Magic_h

#include <Pegasus/Common/Config.h>

PEGASUS_NAMESPACE_BEGIN

/** The Magic class implements a convenient way to use magic numbers in
    user-defined classes. Magic numbers help detect use of uninitialized,
    destructed, or corrupted objects.

    To instrument a class to use magic numbers simply add a class member
    of type Magic<> as shown in thsi example:

        \code
        class MyClass
        {
        public:

            MyClass();

            ~MyClass();

            void foo();

        private:
            Magic<0xC531B144> _magic;
        };
        \endcode

    Choose whatever number you like for a magic number. The number above was
    generated by the Linux uuidgen utility (Windows has a utility with the
    same name).

    To test magic number, add the following expression wherever necessary
    (usually as the first line of every member function).

        \code
        PEGASUS_ASSERT_DEBUG(_magic);
        \endcode

    Here's a typical example:

        \code
        MyClass::~MyClass()
        {
            PEGASUS_ASSERT_DEBUG(_magic);
        }
        \endcode

    This tests whether the magic number is 0xC531B144 and asserts if it is
    not.

    Note that using a magic number makes the user-defined class 4 bytes larger
    but there is no run-time overhead unless you compile with PEGASUS_DEBUG.
    The Magic constructor and destructor are empty without PEGASUS_DEBUG and
    are discarded by the compiler.

    CAUTION: You may be tempted to compile out the magic member when
    PEGASUS_DEBUG undefined. However, this will causes unpredictable
    behavior when debug libraries are mixed with non-debug libraries. The
    structure alignment and size will be different and will lead to crashes.
    This is only safe if a class is internal to a library.
*/
template<Uint32 MAGIC_NUMBER>
class Magic
{
public:

    /** Default constructor. Sets the magic number.
    */
    Magic()
    {
#ifdef PEGASUS_DEBUG
        _magic = MAGIC_NUMBER;
#endif
    }

    /** Destructor. Clears the magic number (with the pattern 0xDDDDDDDD).
    */
    ~Magic()
    {
#ifdef PEGASUS_DEBUG
        _magic = 0xDDDDDDDD;
#endif
    }

    /** Allows magic number to be used as a boolean expression. This function
        tests the magic number and return true if valid.
    */
    operator bool() const
    {
#ifdef PEGASUS_DEBUG
        return _magic == MAGIC_NUMBER;
#else
        return true;
#endif
    }

private:
    Uint32 _magic;
};

PEGASUS_NAMESPACE_END

#endif /* Pegasus_Magic_h */
